﻿using System.Collections.Concurrent;
using System.Diagnostics;
using System.ServiceModel.Dispatcher;
using Curse.Voice.HostInterface;
using Curse.Voice.HostInterface.Models;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.Serialization;
using System.ServiceModel;
using System.Text;
using System.Threading.Tasks;
using System.Threading;

namespace Curse.Voice.Host
{
    public class VoiceSession : IDisposable
    {

        #region Constructor

        private bool _isAlive = true;
        private ConcurrentQueue<VoiceTransmission> _queue = new ConcurrentQueue<VoiceTransmission>();

        public VoiceSession(VoiceInstance parent, VoiceUser user, ICurseVoiceCallbackClient callbackClient)
        {
            Parent = parent;
            User = user;
            CallbackClient = callbackClient;
            ID = callbackClient.GetHashCode();
            CommunicationChannel = callbackClient as ICommunicationObject;
            if (CommunicationChannel == null)
            {
                throw new ArgumentException("callbackClient must be an ICommunicationObject");
            }
            CommunicationChannel.Faulted += CommunicationChannel_Faulted;
            CommunicationChannel.Closed += CommunicationChannel_Faulted;
            IsConnected = true;
#if USE_THREADS
            new Thread(TransmissionThread) { IsBackground = true, Name = "TransmissionThread-" + user.ID }.Start();
#endif
        }

#if USE_THREADS
        public void QueueTransmission(VoiceTransmission transmission)
        {
            _queue.Enqueue(transmission);
        }

        private void TransmissionThread()
        {
            while (_isAlive)
            {
                Thread.Sleep(1);
                VoiceTransmission voiceTransmission = null;
                if (_queue.Count > 0 && _queue.TryDequeue(out voiceTransmission))
                {
                    try
                    {
                        CallbackClient.ReceiveTransmissionSync(voiceTransmission);
                    }
                    catch (Exception ex)
                    {

                        Debug.WriteLine(ex.Message);
                    }

                }
            }
        }
#endif

        #endregion

        #region Properties

        public VoiceInstance Parent { get; private set; }
        public VoiceUser User { get; private set; }
        public int ID { get; private set; }
        public ICurseVoiceCallbackClient CallbackClient { get; private set; }
        public ICommunicationObject CommunicationChannel { get; private set; }

        public bool IsConnected { get; private set; }

        public bool IsCreator
        {
            get { return User.UserID.HasValue && User.UserID.Value == Parent.CreatorUserID; }
        }

        public bool IsOwner
        {
            get { return ID == Parent.OwnerSessionID; }
        }
        #endregion


        void CommunicationChannel_Faulted(object sender, EventArgs e)
        {
            IsConnected = false;
            _isAlive = false;
            Parent.ClientDisconnected(this);
        }


        public async Task NotifyUserJoinedAsync(VoiceUser joinedUser)
        {
            await CallbackClient.ReceiveUserJoinedAsync(joinedUser);
        }

        public async Task NotifyUserLeftAsync(int userID)
        {
            if (!IsConnected)
            {
                return;
            }
            try
            {
                await CallbackClient.ReceiveUserLeftAsync(userID);
            }
            catch (Exception ex)
            {

                Debug.WriteLine("Failed NotifyUserLeftAsync: " + ex.Message);
            }
            
        }

        public async Task Transmit(VoiceTransmission transmission)
        {
            await CallbackClient.ReceiveTransmission(transmission);
        }

        public void TransmitSync(VoiceTransmission transmission)
        {
            CallbackClient.ReceiveTransmissionSync(transmission);
        }

#if USE_THREADS
        public void TransmitSyncQueue(VoiceTransmission transmission)
        {
            QueueTransmission(transmission);
        }
#endif
        public async Task EndTransmit(int userID)
        {
            await CallbackClient.ReceiveEndTransmission(userID);
        }

        public async Task UserUpdate(VoiceUser voiceUser)
        {
            await CallbackClient.ReceiveUserUpdate(voiceUser);
        }

        public async Task SendChatMessage(ChatMessage message)
        {
            await CallbackClient.ReceiveChatMessage(message);
        }

        public void Dispose()
        {

            try
            {
                if (CommunicationChannel != null && CommunicationChannel.State == CommunicationState.Opened)
                {
                    CommunicationChannel.Close(TimeSpan.FromSeconds(5));
                }
            }
            catch
            {

            }


            User = null;
            CallbackClient = null;
            CommunicationChannel = null;
        }

    }
}
