﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Net;
using System.Security.Cryptography.X509Certificates;
using System.ServiceProcess;
using Curse.Logging;
using Curse.Voice.HostManagement;
using System.ServiceModel;
using System.ServiceModel.Description;
using System.Threading;
using Curse.SocketInterface;
using Curse.Voice.Helpers;
using Microsoft.ServiceModel.WebSockets;
using Curse.WebRTC;
using System.Reflection;
using Curse.Friends.ServiceClients;
using Curse.Friends.Statistics;
using Curse.ServiceClients;
using Curse.ServiceEncryption;
using Curse.Voice.Contracts;
using Curse.Voice.HostRuntime.VoiceService;

namespace Curse.Voice.HostRuntime
{
    public partial class CurseVoiceRuntimeService : ServiceBase
    {


        public CurseVoiceRuntimeService()
        {
            InitializeComponent();
        }
        
        private WebSocketHost _voiceServiceHost = null;
        private ServiceHost _managementServiceHost = null;
        
        protected override void OnStart(string[] args)
        {

            var machineKey = MachineKeyHelper.GetMachineKey().ToString();
            var mode = MachineKeyHelper.GetMode();

            VoiceServerLog.Initialize(EventLog, machineKey);
            VoiceServerLog.Info("Voice Server Starting", true, new { HostRuntimeConfiguration.Instance.VoicePortNumbers });

            VoiceServerLog.Info("Initializing encryption tokens...");
            EncryptionToken.Initialize(HostRuntimeConfiguration.Instance.EncryptionKey, HostRuntimeConfiguration.Instance.EncryptionIterations);

            VoiceServerLog.Info("Initializing service clients...");
            FriendsServiceClients.Initialize(HostRuntimeConfiguration.Instance.EncryptionKey, HostRuntimeConfiguration.Instance.EncryptionIterations, HostRuntimeConfiguration.Instance.CentralServiceApiKey);

            // Limit timeouts to 5 seconds, as to not block
            //ServicesConfiguration.DefaultTimeoutSeconds = 5;
            
            var currentDirectoryPath = Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location);
            VoiceServerLog.Info("Setting current directory: " + currentDirectoryPath);
            Directory.SetCurrentDirectory(currentDirectoryPath);

            var registerStatus = RegisterVoiceHostStatus.Error;

            try
            {
                try
                {
                    RTCPeerConnection.Init();
                }
                catch (Exception ex)
                {
                    VoiceServerLog.Exception(ex, "Failed to initialize WebRTC");
                    ex = ex.InnerException;
                    while (ex != null)
                    {
                        VoiceServerLog.Exception(ex);
                        ex = ex.InnerException;
                    }
                    throw;
                }

                // Start the TCP/UDP Server
                VoiceServer.Instance.Initialize(machineKey, HostRuntimeConfiguration.Instance.MaxUsersPerVoiceSession, HostRuntimeConfiguration.Instance.MaxUsersPerVideoSession);
                VoiceServer.Instance.Start(HostRuntimeConfiguration.Instance.VoicePortNumbers.Select(portNumber => new IPEndPoint(IPAddress.Any, portNumber)).ToArray());

                VoiceServerLog.Info("WebSocket opening...");
                SetupWebSocketHost();                
                VoiceServerLog.Info("WebSocket opened");

                // Start the Management Service
                var managementEndpoint = "net.tcp://0.0.0.0:" + HostRuntimeConfiguration.Instance.ManagementPortNumber + "/management-service";
                VoiceServerLog.Info("Management Service Host Starting", true, new { Endpoint = managementEndpoint });

                var hostManagementInstance = new VoiceHostManagementService();
                _managementServiceHost = TcpBindingHelper.CreateServiceHost<IVoiceHostManagement>(hostManagementInstance, managementEndpoint, true);               
                _managementServiceHost.Open();                                               

                VoiceServerLog.Info("Management Service Started");

                // Register With the Central Service
                VoiceServerLog.Info("Registering Host to Central Service...");
                registerStatus = RegisterSelf(machineKey, mode);

                if (registerStatus == RegisterVoiceHostStatus.Error)
                {
                    throw new Exception("Failed to register with central service!");
                }

                if (registerStatus == RegisterVoiceHostStatus.Updating)
                {
                    throw new Exception("Shutting down service, so it can be updated.");
                }

                VoiceServerLog.Info("Registered to central service");
                HostPerformanceMonitor.Initialize();
                FriendsStatsManager.Started();

            }
            catch (Exception ex)
            {
                VoiceServerLog.Exception(ex, "Voice server or management service has failed to start. Attempting graceful shutdown. See log for details.", true);
                StopService(registerStatus == RegisterVoiceHostStatus.Successful);
                Thread.Sleep(1000);
                throw;
            }

            VoiceServerLog.Info("Voice Server Started", true);
        }

        private void SetupWebSocketHost()
        {

#if DISABLE_WS
            VoiceServerLog.Warn("WebSocket disabled.");
            return;
#endif

            var webSocketUri = new Uri("wss://0.0.0.0:443");

            _voiceServiceHost = new WebSocketHost(typeof(VoiceWebSocketService),
                new ServiceThrottlingBehavior
                {
                    MaxConcurrentSessions = int.MaxValue,
                    MaxConcurrentCalls = 20
                }, webSocketUri);

            VoiceServerLog.Info("Looking for a certificate: " + HostRuntimeConfiguration.Instance.CertificateSubject);

            var certificate = FindCertificate(HostRuntimeConfiguration.Instance.CertificateSubject);

            if (certificate != null)
            {
                _voiceServiceHost.Credentials.ServiceCertificate.Certificate = certificate;
                VoiceServerLog.Info("WebSocket service will be bound to certificate: " + certificate.SubjectName.Name);
            }
            else
            {
                VoiceServerLog.Error("Failed to bind service to a certificate.");
            }

            VoiceServerLog.Info("Opening WebSocket host...");
            var binding = WebSocketHost.CreateWebSocketBinding(https: true, sendBufferSize: SocketConstants.BufferSize, receiveBufferSize: SocketConstants.BufferSize);
            binding.SendTimeout = TimeSpan.FromMilliseconds(500);
            binding.OpenTimeout = TimeSpan.FromDays(1);
            _voiceServiceHost.AddWebSocketEndpoint(binding);
            _voiceServiceHost.Open();

            VoiceWebSocketService.StartCleanupThread();
        }

        private X509Certificate2 FindCertificate(string friendlyName)
        {
            var namesFound = new List<string>();
            foreach (var storeLocation in (StoreLocation[])Enum.GetValues(typeof(StoreLocation)))
            {
                foreach (var storeName in (StoreName[]) Enum.GetValues(typeof(StoreName)))
                {
                    var store = new X509Store(storeName, storeLocation);

                    try
                    {
                        store.Open(OpenFlags.OpenExistingOnly);
                        var certificates = store.Certificates.Cast<X509Certificate2>();
                        namesFound.AddRange(certificates.Select(p => p.FriendlyName + ": " + p.Subject));
                        var certificate = certificates.FirstOrDefault(p => p.FriendlyName == friendlyName);
                        if (certificate != null)
                        {
                            return certificate;
                        }
                        certificate = certificates.FirstOrDefault(p => p.Subject.Contains("CN=" + friendlyName));
                        if (certificate != null)
                        {
                            return certificate;
                        }
                    }
                    catch
                    {
                        
                    }
                    finally
                    {
                        store.Close();
                    }
                }
            }

            VoiceServerLog.Warn("Failed to find a suitable certification.", new { CertsFound = namesFound.ToArray() });
            return null;
        }

        private RegisterVoiceHostStatus RegisterSelf(string machineKey, int mode)
        {
            VoiceServerLog.Info("Registering with central service at: " + HostRuntimeConfiguration.Instance.CentralServiceUrl);

            
            for (var i = 0; i < 10; i++)
            {
                try
                {
                    using (var client = new CurseVoiceServiceClient(BindingHelper.GetBasicHttpBinding(30), new EndpointAddress(HostRuntimeConfiguration.Instance.CentralServiceUrl)))
                    {

                        var resp = client.RegisterVoiceHostV2(HostRuntimeConfiguration.Instance.CentralServiceApiKey, machineKey, Environment.MachineName, AssemblyHelper.GetApplicationVersion(), (VoiceInstanceMode)mode);
                        
                        VoiceServer.Instance.UpdateConfiguration(resp.Configuration.MaxUsersPerVoiceSession, resp.Configuration.MaxUsersPerVideoSession);

                        switch (resp.Status)
                        {
                                case RegisterVoiceHostStatus.Successful:
                                    VoiceServerLog.Info("Setting Minimum Client Version to " + resp.MinimumClientVersion);
                                    VoiceServer.Instance.UpdateMinimumClientVersion(resp.MinimumClientVersion);
                                    VoiceServerLog.Info("Registration Completed");

                                    VoiceServerLog.Info("Initializing stats manager...");
                                    FriendsStatsManager.Initialize(1, "VoIP", resp.DataRegionID);
                                    FriendsStatsManager.BeginStartup();
                                    break;

                                case RegisterVoiceHostStatus.Updating:
                                    VoiceServerLog.Info("Voice host will be updated to a new version.");
                                    break;                                
                                default:
                                    VoiceServerLog.Info("Voice host registration failed: " +  resp.Status);
                                    break;

                        }
                        
                        return resp.Status;
                        
                    }
                }
                catch (Exception ex)
                {
                    VoiceServerLog.Exception(ex, "Failed to connect to central service after " + (i + 1) + " attempts.");
                }

            }

            return RegisterVoiceHostStatus.Error;
        }

        protected override void OnShutdown()
        {
            VoiceServerLog.Info("Windows Shutting Down");
            StopService();
        }

        protected override void OnStop()
        {
            StopService();
        }

        public void OnDebug()
        {
            OnStart(null);

        }

        public void OnDebugStop()
        {
            OnStop();
        }


        private void StopService(bool issueShutdownCommand = true)
        {
            try
            {
                FriendsStatsManager.BeginShutdown();
                Logger.Info("Requesting Additional Time");
                RequestAdditionalTime(1000 * 30); // 30 Seconds
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to request additional time!");
            }

            // Put the voice server in a state which rejects new connections
            try
            {
                Logger.Info("Service Stopping");

                if (_voiceServiceHost != null)
                {
                    _voiceServiceHost.Close();
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex);
            }

            try
            {
                if (issueShutdownCommand)                {
                    // Shutdown this host at the central server side
                    Logger.Info("Sending Shutdown Command to Cental Service...");


                    using (var client = new CurseVoiceServiceClient(new BasicHttpBinding(), new EndpointAddress(HostRuntimeConfiguration.Instance.CentralServiceUrl)))
                    {
                        var resp = client.ShutdownVoiceHost(HostRuntimeConfiguration.Instance.CentralServiceApiKey, MachineKeyHelper.GetMachineKey().ToString());

                        if (resp.Status == ShutdownHostStatus.Successful) // Failover was successful, let's let the connected clients know!
                        {
                            if (resp.FailoverInstructions != null && resp.FailoverInstructions.Any())
                            {
                                Logger.Info("Shutdown command completed. Failing over " + resp.FailoverInstructions.Length.ToString("###,##0") + " instances...");
                                VoiceServer.Instance.Failover(resp.FailoverInstructions);
                                Thread.Sleep(5000);
                                // Give up to two seconds for clients to receive failover notification
                                Logger.Info("Instance failover has completed successfully!");
                            }
                        }
                        else
                        {
                            Logger.Info("Shutdown command failed with status '" + resp.Status + ". Failover will not occur.");
                        }
                    }
                }
                else
                {
                    Logger.Warn("Skipping failover");
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to notify central service of shutdown!");
            }


            try
            {
                if (_managementServiceHost != null)
                {
                    Logger.Info("Stopping management WCF service...");

                    _managementServiceHost.Abort();
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to stop management service host: " + ex.Message);
            }


            try
            {
                if (VoiceServer.Instance != null)
                {
                    Logger.Info("Stopping Voice Server with " + VoiceServer.Instance.ShutdownInstanceCount.ToString("###,##0") + " Voice Instance(s)");
                    VoiceServer.Instance.Stop();
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to stop voice server: " + ex.Message);
            }


            try
            {
                FriendsStatsManager.Shutdown();
            }
            catch (Exception ex)
            {
                Logger.Error(ex);
            }
        }
    }
}
