﻿using System;
using System.Collections.Concurrent;
using System.Threading;
using System.Threading.Tasks;
using Curse.Friends.ServiceClients;
using Curse.Logging;
using Curse.ServiceClients.Contracts;

namespace Curse.Voice.HostRuntime
{
    public static class GroupCallbackManager
    {
        private static readonly object _syncRoot = new object();
        private static bool _isInitalized;
        private static bool _isAlive = true;
        private static readonly LogCategory Logger = new LogCategory("GroupCallbackManager");
        private static readonly ConcurrentQueue<ChangeRequestQueueItem> _voiceSessionChanges = new ConcurrentQueue<ChangeRequestQueueItem>();

        public static void Initialize()
        {
            lock (_syncRoot)
            {
                if (_isInitalized)
                {
                    return;
                }

                Task.Factory.StartNew(ProcessQueue, TaskCreationOptions.LongRunning);

                _isInitalized = true;
            }            
        }

        public static void Shutdown()
        {
            _isAlive = false;
        }

        private class ChangeRequestQueueItem
        {
            public ChangeRequestQueueItem(string sessionCode, Guid groupID, int[] userIDs, VoiceHostCallbackType changeType)
            {
                Content = new VoiceHostCallbackRequest
                {
                    GroupID = groupID,
                    UserIDs = userIDs,
                    ChangeType = changeType,
                    InviteCode = sessionCode
                };
            }

            public VoiceHostCallbackRequest Content { get; private set; }
            public int Attempts { get; set; }
            public bool HasExpired
            {
                get { return Attempts >= 3; }
            }
        }

        public static void UserJoined(string sessionCode, Guid groupID, int userID)
        {
            QueueCallChanged(sessionCode, groupID, new[] { userID }, VoiceHostCallbackType.UsersJoined);
        }

        public static void UserLeft(string sessionCode, Guid groupID, int userID)
        {
            QueueCallChanged(sessionCode, groupID, new []{userID}, VoiceHostCallbackType.UsersLeft);
        }

        public static void CallEnded(string sessionCode, Guid groupID, int[] userIDs)
        {
            QueueCallChanged(sessionCode, groupID, userIDs, VoiceHostCallbackType.CallEnded);
        }

        public static void CallStarted(string sessionCode, Guid groupID, int[] userIDs)
        {
            QueueCallChanged(sessionCode, groupID, userIDs, VoiceHostCallbackType.CallStarted);
        }

        private static void QueueCallChanged(string sessionCode, Guid groupID, int[] userIDs, VoiceHostCallbackType changeType)
        {
            try
            {
                _voiceSessionChanges.Enqueue(new ChangeRequestQueueItem(sessionCode, groupID, userIDs, changeType));

            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to enqueue group change request!");
            }
            
        }

        private static void ProcessQueue()
        {
            while (_isAlive)
            {
                try
                {
                    ChangeRequestQueueItem item;
                    while (_isAlive && _voiceSessionChanges.TryDequeue(out item))
                    {
                        if (MakeCallback(item))
                        {
                            continue;
                        }

                        if (item.HasExpired)
                        {
                            Logger.Warn("Failed to make voice session change callback after max attempts!",
                                new {item.Content});
                        }
                        else
                        {
                            _voiceSessionChanges.Enqueue(item);
                        }
                    }

                    Thread.Sleep(100);
                }
                catch(ThreadAbortException) { }
                catch (Exception ex)
                {
                    Logger.Info(ex, "Queue thread is exiting...");
                    return;                    
                }                
            }
        }

        private static bool MakeCallback(ChangeRequestQueueItem item)
        {            
            try
            {
                ++item.Attempts;
                FriendsServiceClients.Instance.CallsAdmin.VoiceHostCallback(item.Content);
                return true;
            }
            catch (Exception ex)
            {
                Logger.Trace(ex, string.Format("Failed attempt {0} to make voice session change callback!", item.Attempts), new {item.Content});
                return false;
            }
        }
    }
}
