﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.ServiceModel;
using Curse.Friends.Statistics;
using Curse.Logging;
using Curse.SocketServer;
using Curse.Voice.Contracts;
using Curse.Voice.HostManagement;
using Curse.Voice.HostManagement.Extensions;
using Curse.Voice.HostManagement.Models;
using Curse.Voice.HostManagement.Responses;
using Curse.Voice.HostManagement.Exceptions;

namespace Curse.Voice.HostRuntime
{
    [ServiceBehavior(AddressFilterMode = AddressFilterMode.Any, ConcurrencyMode = ConcurrencyMode.Multiple, InstanceContextMode = InstanceContextMode.Single, UseSynchronizationContext = false)]
    public class VoiceHostManagementService : IVoiceHostManagement
    {

        private static readonly LogCategory Logger = new LogCategory("VoiceHostManagementService") { AlphaLevel = LogLevel.Trace };      

        public ProvisionInstanceResponse ProvisionGroupVoiceInstance(string instanceIdentifier, Guid groupID, string joinCode, string apiKey)
        {
            return DoProvisionVoiceInstance(apiKey, instanceIdentifier, joinCode, VoiceInstanceType.Group, VoiceInstanceMode.Audio, null, groupID);
        }
     
        public ProvisionInstanceResponse ProvisionTypedVoiceInstance(string instanceIdentifier, int ownerUserID, VoiceInstanceType type, string apiKey)
        {
            return DoProvisionVoiceInstance(apiKey, instanceIdentifier, null, type, VoiceInstanceMode.Audio, ownerUserID, null);
        }
       
        public ProvisionInstanceResponse ProvisionGroupVoiceInstanceV2(string apiKey, string instanceIdentifier, Guid groupID, string joinCode, VoiceInstanceMode mode)
        {
            return DoProvisionVoiceInstance(apiKey, instanceIdentifier, joinCode, VoiceInstanceType.Group, mode, null, groupID);
        }
        
        public ProvisionInstanceResponse ProvisionTypedVoiceInstanceV2(string apiKey, string instanceIdentifier, int ownerUserID, VoiceInstanceType type, VoiceInstanceMode mode)
        {
            return DoProvisionVoiceInstance(apiKey, instanceIdentifier, null, type, mode, ownerUserID, null);
        }

        public UnlockInstanceResponse UnlockVoiceInstance(string instanceIdentifier, string apiKey)
        {
            var ipAddress = OperationContext.Current.GetClientIPAddress().ToString();
            if (apiKey != HostRuntimeConfiguration.Instance.CentralServiceApiKey)
            {
                Logger.Warn("Unlock Instance: Attempt to access API protected method", new { IPAddress = ipAddress });
                return null;
            }

            try
            {
                var instance = VoiceServer.Instance.GetInstance(instanceIdentifier);

                if (instance == null)
                {
                    Logger.Warn("Unable to unlock instance. It does not exist on the server.");
                    return new UnlockInstanceResponse { Status = UnlockInstanceStatus.Error };
                }

                instance.ChangeCallType(VoiceInstanceType.MultiFriend);
                return new UnlockInstanceResponse { Status = UnlockInstanceStatus.Successful };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unlock Instance: Unable to change the instance to an ad-hoc session.");
                return new UnlockInstanceResponse { Status = UnlockInstanceStatus.Error };
            }
        }

        private ProvisionInstanceResponse DoProvisionVoiceInstance(string apiKey, string instanceIdentifier, string joinCode, VoiceInstanceType type, VoiceInstanceMode mode, int? userID, Guid? groupID)
        {
            var ipAddress = OperationContext.Current.GetClientIPAddress().ToString();
            if (apiKey != HostRuntimeConfiguration.Instance.CentralServiceApiKey)
            {
                Logger.Warn("Provision Instance: Attempt to access API protected method", new { IPAddress = ipAddress });
                return null;
            }

            Logger.Debug("Provision Instance: Received request to provision a new instance.",
                new
                {
                    Identifier = instanceIdentifier,
                    GroupID = groupID,
                    UserID = userID,
                    IPAddress = ipAddress
                });

            try
            {
                VoiceServer.Instance.StartNewInstance(instanceIdentifier, joinCode, type, mode, userID, groupID);
            }
            catch (ProvisionInstanceException ex)
            {
                Logger.Error(ex, "Provision Instance: Failed due to error.");
                return new ProvisionInstanceResponse(ex.Status);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Provision Instance: Failed due to unhandled exception.");
                return new ProvisionInstanceResponse(ProvisionInstanceStatus.FailedUnhandledException);
            }

            return new ProvisionInstanceResponse(ProvisionInstanceStatus.Successful);
        }

        public KickUsersResponse KickUsers(string instanceIdentifier, int kickerUserID, int[] userIDsToKick, string apiKey)
        {
            switch (RemoveUsers(instanceIdentifier, kickerUserID, userIDsToKick, UserDisconnectReason.Kicked, apiKey).Status)
            {
                case RemoveUsersStatus.Successful:
                    return new KickUsersResponse {Status = KickUsersStatus.Successful};
                default:
                    return new KickUsersResponse {Status = KickUsersStatus.Error};
            }
        }

        public RemoveUsersResponse RemoveUsers(string instanceIdentifier, int requestorUserID, int[] userIDsToRemove, UserDisconnectReason reason, string apiKey)
        {
            if (apiKey != HostRuntimeConfiguration.Instance.CentralServiceApiKey)
            {
                Logger.Warn("Attempt to access API protected method", new { IPAddress = OperationContext.Current.GetClientIPAddress().ToString() });
                return null;
            }

            try
            {
                var instance = VoiceServer.Instance.GetInstance(instanceIdentifier);
                if(instance==null)
                {
                    return new RemoveUsersResponse{Status = RemoveUsersStatus.Error};   
                }

                var requestorID = 0;
                var requestor = instance.GetSessionByUserID(requestorUserID);
                if (requestor != null)
                {
                    requestorID = requestor.ID;
                }

                foreach (var id in userIDsToRemove)
                {
                    var userToRemove = instance.GetSessionByUserID(id);
                    if (userToRemove == null)
                    {
                        instance.RemovePendingUser(id, reason == UserDisconnectReason.LeftGroup ? RemovePendingUserReason.LeftGroup : RemovePendingUserReason.Kicked);
                        continue;
                    }

                    instance.DisconnectUser(requestorID, userToRemove.ID, reason);
                }

                return new RemoveUsersResponse { Status = RemoveUsersStatus.Successful };
            }
            catch (Exception ex)
            {
                VoiceServerLog.Exception(ex);
                return new RemoveUsersResponse { Status = RemoveUsersStatus.Error };
            }
        }

        public AddPendingUsersResponse AddPendingUsers(string instanceIdentifier, PendingUserRequest[] pendingUsers, string apiKey)
        {
            if (apiKey != HostRuntimeConfiguration.Instance.CentralServiceApiKey)
            {
                Logger.Warn("Attempt to access API protected method",
                    new { IPAddress = OperationContext.Current.GetClientIPAddress().ToString() });
                return new AddPendingUsersResponse { Status = AddPendingUsersStatus.Forbidden };
            }

            try
            {
                var instance = VoiceServer.Instance.GetInstance(instanceIdentifier);

                if (instance == null)
                {
                    return new AddPendingUsersResponse { Status = AddPendingUsersStatus.NotFound };
                }


                var successes = new List<PendingUser>();
                foreach (var request in pendingUsers)
                {
                    var pendingUser = instance.AddPendingUser(request.UserID, request.DisplayName, request.AvatarUrl);
                    if (pendingUser != null)
                    {
                        successes.Add(pendingUser);
                    }
                }

                if (successes.Any())
                {
                    instance.BroadcastPendingUsers(successes.ToArray());
                }

                return new AddPendingUsersResponse { Status = AddPendingUsersStatus.Success };
            }
            catch (Exception ex)
            {
                VoiceServerLog.Exception(ex);
                return new AddPendingUsersResponse { Status = AddPendingUsersStatus.Error };
            }

        }

        public RemovePendingUserResponse RemovePendingUser(string instanceIdentifier, int curseUserID, string apiKey)
        {
            if (apiKey != HostRuntimeConfiguration.Instance.CentralServiceApiKey)
            {
                Logger.Warn("Attempt to access API protected method", new {IPAddress = OperationContext.Current.GetClientIPAddress().ToString()});
                return new RemovePendingUserResponse {Status = RemovePendingUserStatus.Forbidden};
            }

            try
            {
                var instance = VoiceServer.Instance.GetInstance(instanceIdentifier);

                if (instance == null)
                {
                    return new RemovePendingUserResponse {Status = RemovePendingUserStatus.NotFound};
                }

                instance.RemovePendingUser(curseUserID, RemovePendingUserReason.Declined);
                return new RemovePendingUserResponse {Status = RemovePendingUserStatus.Successful};
            }
            catch (Exception ex)
            {
                VoiceServerLog.Exception(ex);
                return new RemovePendingUserResponse {Status = RemovePendingUserStatus.Error};
            }
        }

        public DestoryInstanceResponse DestroyVoiceInstance(string instanceIdentifier, string apiKey)
        {
            if (apiKey != HostRuntimeConfiguration.Instance.CentralServiceApiKey)
            {
                Logger.Warn("Attempt to access API protected method", new { IPAddress = OperationContext.Current.GetClientIPAddress().ToString() });
                return null;
            }

            try
            {
                VoiceServer.Instance.DestroyInstance(instanceIdentifier);
            }
            catch (DestroyInstanceException ex)
            {
                VoiceServerLog.Exception(ex);
                return new DestoryInstanceResponse(ex.Status);
            }
            catch (Exception ex)
            {
                VoiceServerLog.Exception(ex);
                return new DestoryInstanceResponse(DestoryInstanceStatus.FailedUnhandledException);
            }

            return new DestoryInstanceResponse(DestoryInstanceStatus.Successful);
        }

        public StatisticsResponse GetStatistics(string apiKey)
        {
            if (apiKey != HostRuntimeConfiguration.Instance.CentralServiceApiKey)
            {
                Logger.Warn("Attempt to access API protected method", new { IPAddress = OperationContext.Current.GetClientIPAddress().ToString() });
                return null;
            }

            try
            {

                VoiceServer.Instance.UpdateCallbackClient(OperationContext.Current);

                return new StatisticsResponse()
                {
                    ActiveConnections = VoiceServer.Instance.ConnectionCount,
                    CurrentBandwidthUsage = 0,
                    TotalSessions = VoiceServer.Instance.TotalInstanceCount,
                    MultiUserSessions = VoiceServer.Instance.ActiveInstanceCount,
                    ActiveSessions = VoiceServer.Instance.TotalInstanceCount,
                    ReceivedBitsPerSecond = HostPerformanceMonitor.ReceivedBitsPerSecond,
                    SentBitsPerSecond = HostPerformanceMonitor.SentBitsPerSecond,
                    ProcessorUtilization = HostPerformanceMonitor.ProcessorUtilization,
                    MemoryUsedKilobytes = HostPerformanceMonitor.MemoryUsageKilobytes
                };
            }
            catch (Exception ex)
            {
                VoiceServerLog.Exception(ex, "Call to GetStatistics failed!");
                return null;
            }

        }

        public InstanceStatisticsResponse GetInstanceStatistics(string apiKey)
        {
            if (apiKey != HostRuntimeConfiguration.Instance.CentralServiceApiKey)
            {
                Logger.Warn("Attempt to access API protected method", new { IPAddress = OperationContext.Current.GetClientIPAddress().ToString() });
                return null;
            }

            try
            {
                VoiceServer.Instance.UpdateCallbackClient(OperationContext.Current);

                var response = new InstanceStatisticsResponse()
                    {
                        Statistics = VoiceServer.Instance.GetInstanceStatistics()
                    };
                return response;
            }
            catch (Exception ex)
            {
                VoiceServerLog.Exception(ex, "Call to GetInstanceStatistics failed!");
                return null;
            }
        }

        public ConnectionDiagnosticsResponse GetConnectionDiagnostics(string apiKey)
        {
            if (apiKey != HostRuntimeConfiguration.Instance.CentralServiceApiKey)
            {                
                Logger.Warn("Attempt to access API protected method", new { IPAddress = OperationContext.Current.GetClientIPAddress().ToString() });
                return null;
            }

            try
            {
                VoiceServer.Instance.UpdateCallbackClient(OperationContext.Current);

                var response = new ConnectionDiagnosticsResponse()
                {
                    Diagnostics = VoiceServer.Instance.GetConnectionDiagnostics()
                };

                return response;
            }
            catch (Exception ex)
            {
                VoiceServerLog.Exception(ex, "Call to GetInstanceStatistics failed!");
                return null;
            }
        }

        public FailoverServerResponse Failover(string apiKey, FailoverInstructions[] failovers)
        {
            if (apiKey != HostRuntimeConfiguration.Instance.CentralServiceApiKey)
            {
                Logger.Warn("[Failover] Attempt to access API protected method.", new { IPAddress = OperationContext.Current.GetClientIPAddress().ToString() });
                return new FailoverServerResponse { Status = FailoverServerStatus.Unauthorized };
            }

            if (VoiceServer.Instance.State == SocketServerState.FailingOver)
            {
                Logger.Warn("Attempt to failover server already in failover state.");
                return new FailoverServerResponse { Status = FailoverServerStatus.InvalidState };
            }

            DateTime startTime = DateTime.UtcNow;
            Logger.Info("Failover started.");

            try
            {
                VoiceServer.Instance.Failover(failovers);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to complete failover.");
                return new FailoverServerResponse { Status = FailoverServerStatus.Error, StatusMessage = ex.Message };
            }

            return new FailoverServerResponse { Status = FailoverServerStatus.Successful, Duration = DateTime.UtcNow - startTime };
        }

        public FailoverServerResponse FailoverInstance(string apiKey, FailoverInstructions failover)
        {
            if (apiKey != HostRuntimeConfiguration.Instance.CentralServiceApiKey)
            {
                Logger.Warn("[FailoverInstance] Attempt to access API protected method.", new { IPAddress = OperationContext.Current.GetClientIPAddress().ToString() });
                return new FailoverServerResponse { Status = FailoverServerStatus.Unauthorized };
            }

            var startTime = DateTime.UtcNow;

            try
            {
                Logger.Debug("Failing over instance: " + failover.InstanceCode);
                VoiceServer.Instance.Failover(failover);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to complete failover for instance: " + failover.InstanceCode);
                return new FailoverServerResponse { Status = FailoverServerStatus.Error, StatusMessage = ex.Message };
            }

            return new FailoverServerResponse { Status = FailoverServerStatus.Successful, Duration = DateTime.UtcNow - startTime };
        }

        public UpdateUserPermissionsResponse UpdateUserPermissions(string instanceIdentifier, VoiceUserPermissions[] userPermissions, string apiKey)
        {
            if (apiKey != HostRuntimeConfiguration.Instance.CentralServiceApiKey)
            {
                Logger.Warn("[UpdateUserPermissions] Attempt to access API protected method.", new { IPAddress = OperationContext.Current.GetClientIPAddress().ToString() });
                return new UpdateUserPermissionsResponse{Status = UpdateUserPermissionsStatus.Unauthorized};
            }

            try
            {
                var instance = VoiceServer.Instance.GetInstance(instanceIdentifier);

                if (instance == null)
                {
                    Logger.Warn("[UpdateUserPermissions] Unable to update permissions. The instance was not found: " + instanceIdentifier);
                    return new UpdateUserPermissionsResponse {Status = UpdateUserPermissionsStatus.Error};
                }

                Logger.Trace("Permissions updated for instance: " + instanceIdentifier);
                instance.UpdateUserPermissions(userPermissions);
                return new UpdateUserPermissionsResponse {Status = UpdateUserPermissionsStatus.Successful};
            }
            catch (Exception ex)
            {
                VoiceServerLog.Exception(ex);
                return new UpdateUserPermissionsResponse {Status = UpdateUserPermissionsStatus.Error};
            }
        }

        public MuteUserResponse MuteUser(string instanceIdentifier, int requestorUserID, int mutedUserID, bool mute, string apiKey)
        {
            if (apiKey != HostRuntimeConfiguration.Instance.CentralServiceApiKey)
            {
                Logger.Warn("Attempt to access API protected method", new { IPAddress = OperationContext.Current.GetClientIPAddress().ToString()});
                return null;
            }

            try
            {
                var instance = VoiceServer.Instance.GetInstance(instanceIdentifier);
                if (instance == null)
                {
                    return new MuteUserResponse {Status = MuteUserStatus.Error};
                }

                instance.ModMuteUser(mutedUserID, mute);
                return new MuteUserResponse {Status = MuteUserStatus.Successful};
            }
            catch (Exception ex)
            {
                VoiceServerLog.Exception(ex);
                return new MuteUserResponse {Status = MuteUserStatus.Error};
            }
        }

        public DeafenUserResponse DeafenUser(string instanceIdentifier, int requestorUserID, int deafenedUserID, bool deafen, string apiKey)
        {
            if (apiKey != HostRuntimeConfiguration.Instance.CentralServiceApiKey)
            {
                Logger.Warn("Attempt to access API protected method", new { IPAddress = OperationContext.Current.GetClientIPAddress().ToString() });
                return null;
            }

            try
            {
                var instance = VoiceServer.Instance.GetInstance(instanceIdentifier);
                if (instance == null)
                {
                    return new DeafenUserResponse { Status = DeafenUserStatus.Error };
                }

                instance.ModDeafenUser(deafenedUserID, deafen);
                return new DeafenUserResponse { Status = DeafenUserStatus.Successful };
            }
            catch (Exception ex)
            {
                VoiceServerLog.Exception(ex);
                return new DeafenUserResponse { Status = DeafenUserStatus.Error };
            }
        }

        public InstanceDetailsResponse GetInstanceDetails(string apiKey, Guid instanceIdentifier)
        {
            if (apiKey != HostRuntimeConfiguration.Instance.CentralServiceApiKey)
            {
                Logger.Warn("Attempt to access API protected method", new { IPAddress = OperationContext.Current.GetClientIPAddress().ToString() });
                return null;
            }

            try
            {
                VoiceServer.Instance.UpdateCallbackClient(OperationContext.Current);

                var instance = VoiceServer.Instance.GetInstance(instanceIdentifier.ToString());

                if (instance == null)
                {
                    return new InstanceDetailsResponse { Status = InstanceDetailsResponseStatus.NotFound };
                }

                instance.DateLastAccessed = DateTime.UtcNow;
                
                return new InstanceDetailsResponse
                {
                    Status = InstanceDetailsResponseStatus.Successful,
                    UserCount = instance.UserCount,
                    DateCreated = instance.DateCreated,
                    GroupID = instance.GroupID,
                    JoinCode = instance.JoinCode
                };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to determine if a group instance is valid!");
                return new InstanceDetailsResponse();
            }
        }

        public bool UpdateConfiguration(string apiKey, VoiceHostConfigurationContract configuration)
        {
            if (apiKey != HostRuntimeConfiguration.Instance.CentralServiceApiKey)
            {
                Logger.Warn("Attempt to access API protected method", new { IPAddress = OperationContext.Current.GetClientIPAddress().ToString() });
                return false;
            }

            VoiceServer.Instance.UpdateConfiguration(configuration.MaxUsersPerVoiceSession, configuration.MaxUsersPerVideoSession);
            return true;
        }
    }
}
