﻿using Curse.Logging;
using Curse.SocketInterface;
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using Curse.SocketServer;
using Curse.Voice.Contracts;
using Curse.Voice.Helpers;
using Curse.Voice.HostManagement.Models;
using Curse.WebRTC;
using UserDisconnectReason = Curse.Voice.Contracts.UserDisconnectReason;


namespace Curse.Voice.HostRuntime
{
    public class VoiceInstance
    {
        private static readonly LogCategory Logger = new LogCategory("VoiceInstance") { Throttle = TimeSpan.FromSeconds(30) };

        public readonly DateTime DateCreated = DateTime.UtcNow;
        public DateTime DateLastAccessed = DateTime.UtcNow;
        public string Identifier { get; private set; }
        public string JoinCode { get; private set; }
        public int CreatorUserID { get; private set; }
        public int OwnerSessionID { get; private set; }
        public Guid? GroupID { get; private set; }
        public bool IsFailingOver { get; set; }
        public Guid UniqueID { get; private set; }

        // Used to prevent ProcessCallEnded from being called more than once.
        private volatile bool _isEnding = false;

        public bool HasOtherJoined { get; set; }
        public bool RequiresAccessToken
        {
            get { return Type == VoiceInstanceType.Friend || Type == VoiceInstanceType.Group; }
        }

        public VoiceInstanceType Type { get; private set; }

        public VoiceInstanceMode Mode { get; private set; }

        private int _memberIDCounter = 0;

        public Int64 InboundVoiceDataCounter { get; set; }
        public Int64 OutboundVoiceDataCounter { get; set; }

        public Int64 InboundVideoDataCounter { get; set; }
        public Int64 OutboundVideoDataCounter { get; set; }

        public ConcurrentDictionary<int, VoiceSession> SessionsByClientID { get; private set; }
        public ConcurrentDictionary<int, VoiceSession> SessionsByUserID { get; private set; }
        public ConcurrentDictionary<int, long> AccessTokenHistory { get; private set; }
        public ConcurrentDictionary<int, bool> ModMutedUsers { get; private set; }
        public ConcurrentDictionary<int, bool> ModDeafenedUsers { get; private set; }

        public ConcurrentDictionary<int, VoiceInstancePendingUser> PendingUsersByUserID { get; private set; }
        private int _currentUsersInCall;
        private int _maxUsersInCall;

        public VoiceInstance(string identifer, string joinCode, VoiceInstanceType type, VoiceInstanceMode mode, int? creatorUserID = null, Guid? groupID = null)
        {
            UniqueID = Guid.NewGuid();
            Identifier = identifer;
            JoinCode = joinCode;
            Type = type;
            Mode = mode;

            if (creatorUserID.HasValue)
            {
                CreatorUserID = creatorUserID.Value;
            }

            if (groupID.HasValue)
            {
                GroupID = groupID.Value;
            }

            SessionsByClientID = new ConcurrentDictionary<int, VoiceSession>();
            SessionsByUserID = new ConcurrentDictionary<int, VoiceSession>();
            PendingUsersByUserID = new ConcurrentDictionary<int, VoiceInstancePendingUser>();
            AccessTokenHistory = new ConcurrentDictionary<int, long>();
            ModMutedUsers = new ConcurrentDictionary<int, bool>();
            ModDeafenedUsers = new ConcurrentDictionary<int, bool>();
        }

        public bool IsExpired
        {
            get
            {
                return DateTime.UtcNow - DateCreated > TimeSpan.FromMinutes(2) 
                      && DateTime.UtcNow - DateLastAccessed > TimeSpan.FromMinutes(2)
                      && SessionsByClientID.Count == 0;
            }
        }

        public VoiceSessionMember[] Users
        {
            get { return SessionsByClientID.Values.Select(p => p.VoiceSessionMember).ToArray(); }
        }

        public PendingUser[] PendingUsers
        {
            get { return PendingUsersByUserID.Values.Select(p => p.User).ToArray(); }
        }

        public IEnumerable<VoiceSession> Sessions
        {
            get { return SessionsByClientID.Values; }
        }

        public int[] UserIDs
        {
            get { return SessionsByUserID.Keys.ToArray(); }
        }

        public Int32 UserCount
        {
            get { return SessionsByClientID.Count; }
        }

        public VoiceSession GetSessionByUserID(int userID)
        {
            VoiceSession found;
            if (SessionsByUserID.TryGetValue(userID, out found))
            {
                return found;
            }

            return null;
        }

        public VoiceSession CreateSession(int? userID, JoinSessionRequest request, ISocketInterface socket, bool? isModMuted, bool? isModDeafened, VoiceUserPermissions permissions)
        {
            var voiceSessionMember = new VoiceSessionMember()
            {
                AvatarUrl = request.AvatarUrl,
                DisplayName = request.DisplayName,
                ClientID = Interlocked.Increment(ref _memberIDCounter),
                UserID = userID,
                InGameName = request.InGameName,
                InGameRegion = request.InGameRegion,
                CodecInfo = request.CodecInfo,
                CanSpeak = permissions.CanSpeak,
                VideoCodec = request.VideoCodec,
            };

            if (isModMuted.HasValue && userID.HasValue)
            {
                if (isModMuted.Value)
                {
                    ModMutedUsers.TryAdd(userID.Value, true);
                    voiceSessionMember.IsModMuted = true;
                }
                else
                {
                    bool value;
                    ModMutedUsers.TryRemove(userID.Value, out value);
                    voiceSessionMember.IsModMuted = false;
                }
            }
            else if (userID.HasValue)
            {
                voiceSessionMember.IsModMuted = ModMutedUsers.ContainsKey(userID.Value);
            }

            if (isModDeafened.HasValue && userID.HasValue)
            {
                if (isModDeafened.Value)
                {
                    ModDeafenedUsers.TryAdd(userID.Value, true);
                    voiceSessionMember.IsModDeafened = true;
                }
                else
                {
                    bool value;
                    ModDeafenedUsers.TryRemove(userID.Value, out value);
                    voiceSessionMember.IsModDeafened = false;
                }
            }
            else if (userID.HasValue)
            {
                voiceSessionMember.IsModDeafened = ModDeafenedUsers.ContainsKey(userID.Value);
            }

            var session = new VoiceSession(this, voiceSessionMember, socket, request.WebRTCType)
            {
                Permissions = permissions,
                SupportsVideo = request.SupportsVideo,
            };

            if (session.IsCreator)
            {
                OwnerSessionID = session.ID;
            }

            AddSession(session);

            Logger.Debug("Voice Session created", new
            {
                permissions,
                isModMuted,
                isModDeafened,
                userID
            });

            return session;
        }

        private void AddSession(VoiceSession session)
        {
            SessionsByClientID.TryAdd(session.ID, session);

            if (!HasOtherJoined && SessionsByClientID.Count > 1)
            {
                HasOtherJoined = true;
            }

            if (session.VoiceSessionMember.UserID.HasValue)
            {
                SessionsByUserID.AddOrUpdate(session.VoiceSessionMember.UserID.Value, session, (a, b) => session);
                RemovePendingUser(session.VoiceSessionMember.UserID.Value);

                //update current and max users in call
                var newCurrent = Interlocked.Increment(ref _currentUsersInCall);
                if (ExchangeIfGreater(ref _maxUsersInCall, newCurrent))
                {
                    VoiceServer.Instance.MaxUsersQueue.Enqueue(new Tuple<string, int>(Identifier, _maxUsersInCall));
                }
            }
        }

        private bool RemoveSession(int clientID)
        {
            VoiceSession removed;
            return RemoveSession(clientID, out removed);
        }

        private bool RemoveSession(int clientID, out VoiceSession removed)
        {
            var successful = SessionsByClientID.TryRemove(clientID, out removed);

            if (!successful)
            {
                return successful;
            }

            if (removed.VoiceSessionMember.UserID.HasValue)
            {
                var id = removed.VoiceSessionMember.UserID.Value;

                VoiceSession removedUserSession;
                SessionsByUserID.TryRemove(id, out removedUserSession);

                // If the session doesn't match the user may have rejoined from a different computer
                if (removedUserSession != null && removedUserSession != removed)
                {
                    SessionsByUserID.TryAdd(id, removedUserSession);
                }
            }

            Interlocked.Decrement(ref _currentUsersInCall);

            return successful;
        }

        public void ChangeCallType(VoiceInstanceType type)
        {
            Type = type;
            GroupID = null;
            BroadcastInstanceChanged(type);
        }

        public void ClientConnected(VoiceSession connectedSession)
        {
            var user = connectedSession.VoiceSessionMember;

            foreach (var session in SessionsByClientID.Values)
            {
                if (!session.IsConnected || session.ID == connectedSession.ID)
                {
                    continue;
                }

                var userJoinedNotification = new UserJoinedNotification()
                {
                    User = user,
                    Timestamp = DateTime.UtcNow
                };

                try
                {
                    session.NotifyUserJoined(userJoinedNotification);
                }
                catch (Exception ex)
                {
                    VoiceServerLog.Exception(ex, "Failed to notify a client of a user joining.");
                }
            }

            if (user.UserID.HasValue)
            {
                HandleUserStatus(true, user.UserID.Value);
            }
        }

        public void ReJoin(VoiceSession currentSession)
        {
            // Remember the previous ID for cleanup later and assign a new ID
            var prevID = currentSession.ID;
            var newID = Interlocked.Increment(ref _memberIDCounter);

            // Assign the new ClientID and update session owner if needed
            currentSession.VoiceSessionMember.ClientID = newID;
            if (OwnerSessionID == prevID)
            {
                OwnerSessionID = newID;
            }

            // Change the ID in the session dictionary
            VoiceSession removed;
            SessionsByClientID.TryRemove(prevID, out removed);
            SessionsByClientID.TryAdd(newID, currentSession);

            // Send a JoinSessionResponse back to the requesting user
            try
            {
                currentSession.ServerSocket.SendContract(new JoinSessionResponse
                {
                    ClientID = currentSession.ServerSocket.ClientID,
                    Status = JoinSessionStatus.Successful,
                    Users = Users,
                    CurrentUserID = currentSession.ID,
                    Type = Type,
                    PendingUsers = PendingUsers,
                    Timestamp = DateTime.UtcNow
                });

                if (currentSession.UseSdp)
                {
                    currentSession.SendOffer();
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to notify client of their new ID");
            }

            // Send a GetUsersReponse back to everyone else. This will re-sync the
            // member list without triggering join and leave notification sounds.
            var users = new GetUsersResponse()
            {
                Users = Users,
                SessionType = Type,
                PendingUsers = PendingUsers,
                Timestamp = DateTime.UtcNow
            };
            foreach (var session in SessionsByClientID.Values)
            {
                try
                {
                    if (!session.IsConnected || session.ID == currentSession.ID)
                    {
                        continue;
                    }

                    session.ServerSocket.SendContract(users);

                    if (session.UseSdp)
                    {
                        session.SendOffer();
                    }
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to notify a client of a user joining.");
                }
            }
        }

        public void ClientDisconnected(VoiceSession disconnectedSession)
        {
            Logger.Trace("Client disconnected", new { Identifier, Type, Mode, ClientID = disconnectedSession.ID });

            // Remove this session from our list
            if (!RemoveSession(disconnectedSession.ID))
            {
                Logger.Warn("Client disconnected and a session could not be removed.", new { Identifier, Type, Mode, ClientID = disconnectedSession.ID });
                return;
            }

            // If this session was the creator, promote someone else
            if (disconnectedSession.IsOwner)
            {
                var firstSession = SessionsByClientID.Values.FirstOrDefault();
                OwnerSessionID = firstSession != null ? firstSession.ID : 0;
            }

            var user = disconnectedSession.VoiceSessionMember;

            // Notify all connected clients            
            var notification = new UserLeftNotification()
            {
                UserID = user.ClientID,
                Timestamp = DateTime.UtcNow
            };

            foreach (var session in SessionsByClientID.Values)
            {
                if (!session.IsConnected)
                {
                    continue;
                }

                try
                {
                    session.NotifyUserLeft(notification);
                }
                catch (Exception ex)
                {
                    VoiceServerLog.Exception(ex, "Failed to notify a client of a user disconnecting.");
                }
            }

            if (user.UserID.HasValue)
            {
                HandleUserStatus(false, user.UserID.Value);
            }

            // If we no longer have any users, and this is a 
            if (ShouldEndAfterLastUser)
            {
                ProcessCallEnded(UserDisconnectReason.CallEnded);
            }
        }

        public void UserLeft(VoiceSession session)
        {
            // Remove the user's access token history            
            RemoveAccessTokenHistory(session.VoiceSessionMember);

            // If this is a friend or multi-friend call, end the call when only one user remains
            if ((Type == VoiceInstanceType.Friend || Type == VoiceInstanceType.MultiFriend) && SessionsByUserID.Count <= 2)
            {
                ProcessCallEnded(UserDisconnectReason.CallEnded);
            }
            else
            {
                DisconnectUser(session.ID, session.ID, UserDisconnectReason.CallEnded);
            }
        }

        public void DisconnectUser(int kicker, int victim, UserDisconnectReason reason)
        {
            VoiceSession victimSession;
            if (!SessionsByClientID.TryGetValue(victim, out victimSession))
            {
                return;
            }

            // If the user was kicked, remove their access token history     
            if (reason == UserDisconnectReason.Kicked)
            {
                RemoveAccessTokenHistory(victimSession.VoiceSessionMember);
            }

            victimSession.SendUserDisconnect(kicker, reason);

            if (reason != UserDisconnectReason.Duplicate)
            {
                BroadcastUserDisconnected(kicker, victim, reason);
            }
        }

        private void ProcessCallEnded(UserDisconnectReason reason)
        {
            Logger.Trace("Call ending", new { reason, Identifier, Type, Mode });

            if (IsFailingOver) // If the instance is failing over, do nothing
            {
                Logger.Trace("Supressing call ended. The instance is failing over.");
                return;
            }

            if (_isEnding) // If the instance is in the middle of processing an ended state, do nothing
            {
                return;
            }

            _isEnding = true;

            var clientIDs = SessionsByClientID.Keys.ToArray();

            foreach (var clientID in clientIDs)
            {
                DisconnectUser(clientID, clientID, reason);
            }

            VoiceServer.Instance.DestroyInstance(Identifier);
        }

        private bool ShouldDisconnectAfterNoAnswer
        {
            get
            {
                // If someone ever joined the call, do not disconnect
                if (HasOtherJoined)
                {
                    return false;
                }

                // If we're still waiting on a user to respond
                if (PendingUsersByUserID.Count > 0)
                {
                    return false;
                }

                // Only disconnect for friend and multifriend calls
                return Type == VoiceInstanceType.Friend || Type == VoiceInstanceType.MultiFriend || Type == VoiceInstanceType.Group;
            }
        }

        private bool ShouldEndAfterLastUser
        {
            get
            {
                // If we're still waiting on a user to respond
                if (PendingUsersByUserID.Count > 0 || SessionsByClientID.Count > 0)
                {
                    return false;
                }

                // Only disconnect for friend and multifriend calls
                return Type == VoiceInstanceType.MultiFriend || Type == VoiceInstanceType.Group;
            }
        }

        public void RemovePendingUser(int userID, RemovePendingUserReason? reason = null)
        {
            if (!PendingUsersByUserID.ContainsKey(userID))
            {
                return;
            }

            VoiceInstancePendingUser removed;
            if (!PendingUsersByUserID.TryRemove(userID, out removed))
            {
                return;
            }

            removed.Remove();

            if (reason.HasValue)
            {
                BroadcastRemovePendingUsers(userID, reason.Value);
            }

            if (ShouldDisconnectAfterNoAnswer)
            {
                ProcessCallEnded(UserDisconnectReason.NoAnswer);
            }
            else if (ShouldEndAfterLastUser)
            {
                ProcessCallEnded(UserDisconnectReason.CallEnded);
            }
        }

        public PendingUser AddPendingUser(int userID, string displayName, string avatarUrl)
        {
            if (userID <= 0 || string.IsNullOrEmpty(displayName))
            {
                Logger.Warn("Unable to add pending user. Invalid data!", new { userID, displayName });
                return null;
            }

            if (SessionsByUserID.ContainsKey(userID))
            {
                Logger.Trace("Unable to add pending user. A member with this user ID already connected.");
                return null;
            }

            if (PendingUsersByUserID.ContainsKey(userID))
            {
                Logger.Trace("Unable to add pending user. A pending user with this user ID has already been added.");
                return null;
            }

            var pendingUser = new PendingUser
            {
                UserID = userID,
                DisplayName = displayName,
                AvatarUrl = avatarUrl
            };

            if (!PendingUsersByUserID.TryAdd(userID, new VoiceInstancePendingUser(this, pendingUser)))
            {
                return null;
            }

            return pendingUser;
        }

        private static bool ExchangeIfGreater(ref int location, int newValue)
        {
            var current = location;

            while (current < newValue)
            {
                // Exchange the value only if it hasn't been changed by another thread
                var previous = Interlocked.CompareExchange(ref location, newValue, current);

                // Return true if the exchange was successful
                if (previous == current)
                {
                    return true;
                }

                // Ensure we're using the latest value
                current = location;
            }

            // Another thread updated the max to a higher value first
            return false;
        }

        #region Access Token

        private void RemoveAccessTokenHistory(VoiceSessionMember member)
        {
            if (member == null || !member.UserID.HasValue)
            {
                return;
            }

            long accessToken;

            AccessTokenHistory.TryRemove(member.UserID.Value, out accessToken);
        }

        private void AddAccessTokenHistory(int userID, long accessToken)
        {
            AccessTokenHistory.AddOrUpdate(userID, accessToken, (i, l) => accessToken);
        }

        public bool CheckAccessToken(int userID, long accessToken)
        {
            // If the token is valid, add it to the access token history
            if (AccessTokenHelper.CheckAccessToken(accessToken, Identifier, userID))
            {
                AddAccessTokenHistory(userID, accessToken);
                return true;
            }


            // If invalid, check the history
            long historicToken;
            if (!AccessTokenHistory.TryGetValue(userID, out historicToken))
            {
                Logger.Warn("Access token is invalid, and no history exists.", new { userID, accessToken });
                return false;
            }

            var success = accessToken == historicToken;

            if (success)
            {
                Logger.Trace("Request access token was invalid, but it matched the user's historical token.", new { userID, accessToken, historicTokenValue = historicToken });
            }
            else
            {
                Logger.Warn("Access token is invalid, and the historic value does not match the supplied value.", new { userID, accessToken, historicTokenValue = historicToken });
            }

            return success;

        }

        #endregion

        #region Broadcasts

        public void BroadcastUserDisconnected(int kicker, int kickedUser, UserDisconnectReason reason)
        {
            var userKicked = new UserDisconnectNotification
            {
                InitiatingUserID = kicker,
                AffectedUserID = kickedUser,
                Reason = reason,
                Timestamp = DateTime.UtcNow
            };

            foreach (var session in SessionsByClientID.Values)
            {
                if (!session.IsConnected)
                {
                    continue;
                }

                try
                {
                    session.NotifyUserDisconnect(userKicked);
                }
                catch (Exception ex)
                {
                    Logger.Debug(ex, "Failed to notify client of a kick broadcast.");
                }
            }
        }

        public void BroadcastInstanceChanged(VoiceInstanceType newType)
        {
            var notification = new VoiceInstanceChangedNotification
            {
                NewType = newType
            };
            foreach (var session in Sessions)
            {
                try
                {
                    session.NotifyInstanceChanged(notification);
                }
                catch (Exception ex)
                {
                    Logger.Debug(ex, "Failed to notify a session of an instance change.");
                }
            }
        }

        public void BroadcastTransmit(VoiceSession sender, ByteBuffer buf)
        {
            var isVoice = RTCRelayServer.IsAudioPacket(buf);
            var headerLen = Srtp.GetHeaderLength(buf);
            var dataLen = buf.Count - headerLen;

            if (isVoice)
            {
                InboundVoiceDataCounter += dataLen;
            }
            else
            {
                InboundVideoDataCounter += dataLen;
            }

            if (!sender.CanSpeak || sender.IsModDeafened)
            {
                // Ignore voice and video if not allowed to speak or listen
                return;
            }

            if (isVoice && (sender.IsSelfMuted || sender.IsModMuted))
            {
                // Only drop voice when self or mod muted
                return;
            }


            foreach (var session in Sessions)
            {
                if (!session.IsConnected)
                {
                    continue;
                }

                if (session.IsModDeafened) // Do not let deafened users hear (or see?)
                {
                    continue;
                }

                if (session.ID == sender.ID && !sender.LoopbackEnabled)
                {
                    continue;
                }

                if (!isVoice && !session.SupportsVideo) // Client doesn't support (or want) video playback
                {
                    continue;
                }

                if (isVoice)
                {
                    OutboundVoiceDataCounter += dataLen;
                }
                else
                {
                    OutboundVideoDataCounter += dataLen;
                }

                try
                {
                    session.SendRtpPacket(sender, buf);
                }
                catch (Exception ex)
                {
                    Logger.Debug(ex, "Failed to notify client of a broadcast");
                }
            }
        }

        public void BroadcastControl(VoiceSession sender, ByteBuffer buf)
        {
            if (!sender.CanSpeak || sender.IsModDeafened)
            {
                // Ignore control messages if not allowed to speak or listen
                return;
            }

            foreach (var session in Sessions)
            {
                if (!session.IsConnected)
                {
                    continue;
                }

                if (session.IsModDeafened) // Do not let deafened users hear (or see?)
                {
                    continue;
                }

                if (session.ID == sender.ID && !sender.LoopbackEnabled)
                {
                    continue;
                }

                if (!session.SupportsVideo) // Client doesn't support (or want) video playback
                {
                    continue;
                }

                try
                {
                    session.SendRtcpPacket(sender, buf);
                }
                catch (Exception ex)
                {
                    Logger.Debug(ex, "Failed to notify client of a control message");
                }
            }
        }

        public void BroadcastUserUpdate(VoiceSessionMember voiceUser)
        {
            var notification = new UserUpdatedNotification()
                {
                    AvatarUrl = voiceUser.AvatarUrl,
                    DisplayName = voiceUser.DisplayName,
                    ID = voiceUser.ClientID,
                    InGameName = voiceUser.InGameName,
                    InGameRegion = voiceUser.InGameRegion,
                    IsModMuted = voiceUser.IsModMuted,
                    IsModDeafened = voiceUser.IsModDeafened,
                    VideoCodec = voiceUser.VideoCodec,
                };


            foreach (var session in SessionsByClientID.Values)
            {
                if (!session.IsConnected)
                {
                    continue;
                }

                try
                {
                    session.NotifyUserUpdated(notification);
                }
                catch (Exception ex)
                {
                    VoiceServerLog.Exception(ex, "Failed to notify a client of a user update.");
                }
            }

        }

        public void BroadcastStartTransmit(VoiceSessionMember voiceUser)
        {
            var notification = new TransmitStartNotification()
            {
                ClientID = voiceUser.ClientID
            };

            foreach (var session in SessionsByClientID.Values)
            {
                if (!session.IsConnected)
                {
                    continue;
                }

                try
                {
                    session.NotifyStartTransmit(notification);
                }
                catch (Exception ex)
                {
                    VoiceServerLog.Exception(ex, "Failed to notify a client of a user starting transmission.");
                }
            }
        }

        public void BroadcastEndTransmit(VoiceSessionMember voiceUser)
        {
            var notification = new TransmitEndNotification()
            {
                ClientID = voiceUser.ClientID
            };

            foreach (var session in SessionsByClientID.Values)
            {
                if (!session.IsConnected)
                {
                    continue;
                }

                try
                {
                    session.NotifyEndTransmit(notification);
                }
                catch (Exception ex)
                {
                    VoiceServerLog.Exception(ex, "Failed to notify a client of a user ending transmission.");
                }
            }
        }

        public void BroadcastFailover(string hostName, string ipAddress, int port, FailoverNotificationReason reason)
        {
            var failoverNotification = new FailoverNotification()
            {
                HostName = hostName,
                IPAddress = ipAddress,
                Port = port,
                Reason = reason
            };

            foreach (var session in SessionsByClientID.Values)
            {
                if (!session.IsConnected)
                {
                    continue;
                }

                try
                {
                    if (RequiresAccessToken)
                    {
                        if (!session.VoiceSessionMember.UserID.HasValue)
                        {
                            Logger.Warn("Unable to failover user. They are missing a user ID.", session.VoiceSessionMember);
                        }
                        else
                        {
                            failoverNotification.AccessToken = AccessTokenHelper.CreateAccessToken(Identifier, session.VoiceSessionMember.UserID.Value);
                        }
                    }

                    session.NotifyFailover(failoverNotification);
                }
                catch (Exception ex)
                {
                    VoiceServerLog.Exception(ex, "Failed to notify a client of a failover.");
                }
            }
        }

        public void BroadcastChatMessage(VoiceSession broadcaster, string message)
        {

            var chatMessageNotification = new ChatMessageNotification()
            {
                SenderID = broadcaster.VoiceSessionMember.ClientID,
                Body = message,
                Timestamp = DateTime.UtcNow
            };

            foreach (var session in SessionsByClientID.Values)
            {
                if (!session.IsConnected)
                {
                    continue;
                }

                try
                {
                    session.NotifyChatMessage(chatMessageNotification);
                }
                catch (Exception ex)
                {
                    VoiceServerLog.Exception(ex, "Failed to notify client of a chat message!");
                }
            }

        }

        public void BroadcastPendingUsers(PendingUser[] pendingUsers)
        {

            var notification = new AddPendingUsersNotification()
            {
                Users = pendingUsers,
            };

            foreach (var session in SessionsByClientID.Values)
            {
                if (!session.IsConnected)
                {
                    continue;
                }

                try
                {
                    session.NotifyPendingUser(notification);
                }
                catch (Exception ex)
                {
                    VoiceServerLog.Exception(ex, "Failed to notify client of a pending user!");
                }
            }

        }

        public void BroadcastRemovePendingUsers(int userID, RemovePendingUserReason reason)
        {

            var notification = new RemovePendingUserNotification
            {
                UserID = userID,
                Reason = reason
            };

            foreach (var session in SessionsByClientID.Values)
            {
                if (!session.IsConnected)
                {
                    continue;
                }

                try
                {
                    session.NotifyRemovePendingUser(notification);
                }
                catch (Exception ex)
                {
                    VoiceServerLog.Exception(ex, "Failed to notify client of a pending user!");
                }
            }

        }

        #endregion

        #region Group Callbacks

        private bool ShouldNotifyGroup
        {
            get { return GroupID.HasValue && VoiceServer.Instance.State == SocketServerState.Started && !IsFailingOver; }
        }

        public void HandleCreated()
        {
            if (!ShouldNotifyGroup)
            {
                return;
            }

            GroupCallbackManager.CallStarted(JoinCode, GroupID.Value, UserIDs);

        }

        public void HandleEnded()
        {
            if (!ShouldNotifyGroup)
            {
                return;
            }

            Logger.Trace("Notifying call service that a call has ended.", new { Identifier });
            GroupCallbackManager.CallEnded(JoinCode, GroupID.Value, UserIDs);

        }

        public void HandleUserStatus(bool joined, int userID)
        {
            if (!ShouldNotifyGroup)
            {
                return;
            }

            if (joined)
            {
                GroupCallbackManager.UserJoined(JoinCode, GroupID.Value, userID);
            }
            else
            {
                GroupCallbackManager.UserLeft(JoinCode, GroupID.Value, userID);
            }
        }

        #endregion

        #region Moderation

        public void UpdateUserPermissions(VoiceUserPermissions[] userPermissions)
        {
            Logger.Debug("Updated user permissions", new { userPermissions });

            foreach (var userPermission in userPermissions)
            {
                var session = GetSessionByUserID(userPermission.UserID);

                if (session == null)
                {
                    // Not connected any more
                    continue;
                }

                var couldSpeak = session.CanSpeak;
                session.CanSpeak = userPermission.CanSpeak;
                session.Permissions = userPermission;
                var hasChanged = couldSpeak != session.CanSpeak
                                 || session.Permissions.CanModDeafen != userPermission.CanModDeafen
                                 || session.Permissions.CanModKick != userPermission.CanModDeafen
                                 || session.Permissions.CanModMute != userPermission.CanModMute;

                if (hasChanged)
                {
                    BroadcastUserUpdate(session.VoiceSessionMember);
                }                
            }
        }

        public void ModMuteUser(int mutedUserID, bool mute)
        {
            DoMuteUser(mutedUserID, mute);
        }

        public bool ModMuteUser(VoiceSession requestorSession, int mutedUserID, bool mute)
        {
            var session = GetSessionByUserID(mutedUserID);
            if (requestorSession.Permissions.CanModMute && CanModerate(requestorSession, session))
            {
                DoMuteUser(mutedUserID, mute);
                return true;
            }
            return false;
        }

        private void DoMuteUser(int userID, bool mute)
        {
            if (mute)
            {
                ModMutedUsers.TryAdd(userID, true);
            }
            else
            {
                bool value;
                ModMutedUsers.TryRemove(userID, out value);
            }

            var mutedUserSession = GetSessionByUserID(userID);
            if (mutedUserSession == null || mutedUserSession.IsModMuted == mute)
            {
                return;
            }

            mutedUserSession.IsModMuted = mute;            
            BroadcastUserUpdate(mutedUserSession.VoiceSessionMember);
        }

        public void ModDeafenUser(int deafenedUserID, bool deafen)
        {
            DoDeafenUser(deafenedUserID, deafen);
        }

        public bool ModDeafenUser(VoiceSession requestorSession, int deafenedUserID, bool deafen)
        {
            var session = GetSessionByUserID(deafenedUserID);
            
            if (requestorSession.Permissions.CanModDeafen && CanModerate(requestorSession, session))
            {
                DoDeafenUser(deafenedUserID, deafen);
                return true;
            }

            return false;
        }

        private void DoDeafenUser(int deafenedUserID, bool deafen)
        {
            if (deafen)
            {
                ModDeafenedUsers.TryAdd(deafenedUserID, true);
            }
            else
            {
                bool value;
                ModDeafenedUsers.TryRemove(deafenedUserID, out value);
            }

            var deafenedUserSession = GetSessionByUserID(deafenedUserID);
            if (deafenedUserSession == null || deafenedUserSession.IsModDeafened == deafen)
            {
                return;
            }

            deafenedUserSession.IsModDeafened = deafen;
            BroadcastUserUpdate(deafenedUserSession.VoiceSessionMember);
        }

        public void ModKickUser(int requestorID, VoiceSession kickedUserSession)
        {
            if (kickedUserSession == null)
            {
                return;
            }

            DisconnectUser(requestorID, kickedUserSession.VoiceSessionMember.ClientID, UserDisconnectReason.Kicked);
        }

        public bool ModKickUser(VoiceSession requestorSession, VoiceSession kickedUserSession)
        {
            if (kickedUserSession == null)
            {
                return true;
            }

            if (requestorSession.Permissions.CanModKick && CanModerate(requestorSession, kickedUserSession))
            {
                DisconnectUser(requestorSession.VoiceSessionMember.ClientID, kickedUserSession.VoiceSessionMember.ClientID, UserDisconnectReason.Kicked);
                return true;
            }
            return false;
        }

        private bool CanModerate(VoiceSession requestorSession, VoiceSession affectedUserSession)
        {
            return affectedUserSession != null && (requestorSession.Permissions.BestRoleRank == 0 || requestorSession.Permissions.BestRoleRank < affectedUserSession.Permissions.BestRoleRank);
        }

        #endregion
    }
}