﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Linq;
using System.Net.Security;
using System.ServiceModel;
using System.ServiceModel.Description;
using System.Text;
using System.Threading.Tasks;
using Curse.Voice.Host;
using Curse.Voice.HostManagement;
using Curse.Voice.HostManagement.Exceptions;
using Curse.Voice.HostManagement.Responses;
using Curse.Voice.HostRuntime.CurseVoiceService;
using System.Threading;
using System.Diagnostics;

namespace Curse.Voice.HostRuntime
{
    public class VoiceInstanceManager : IDisposable
    {
        private bool _isAlive = true;
        public static readonly VoiceInstanceManager Instance = new VoiceInstanceManager();
        private ConcurrentDictionary<string, VoiceInstanceHost> _voiceHostInstances = new ConcurrentDictionary<string, VoiceInstanceHost>();
        public IVoiceHostCallbackClient CallbackClient { get; private set; }
        private int _portNumber;
        private bool _isInitialized = false;

        public int InstanceCount
        {
            get { return _voiceHostInstances.Count; }

        }

        public VoiceInstanceManager()
        {
            new Thread(IdleInstanceCleanup) { IsBackground = true }.Start();

        }

        public void Initialize(int portNumber)
        {            
            _portNumber = portNumber;
            _isInitialized = true;
        }

        public void IdleInstanceCleanup()
        {
            while (_isAlive)
            {
                try
                {
                    var expiredInstances = _voiceHostInstances.Values.Where(p => p.VoiceInstance.IsExpired).ToArray();

                    foreach (var instance in expiredInstances)
                    {
                        if (!_isAlive)
                        {
                            break;
                        }

                        try
                        {
                            DestroyInstance(null, instance.VoiceInstance.Identifier);
                        }
                        catch (Exception ex)
                        {
                            Debug.WriteLine("Unable to destory instance: " + ex.Message);
                        }

                    }
                }
                catch (Exception ex)
                {
                    Debug.WriteLine("IdleInstanceCleanup error: " + ex.Message);
                }

                Thread.Sleep(TimeSpan.FromSeconds(30));
            }
        }

        public void DestroyInstance(IVoiceHostCallbackClient callbackClient, string instanceIdentifier)
        {
            UpdateCallbackClient(callbackClient);

            if (string.IsNullOrEmpty(instanceIdentifier))
            {
                throw new DestoryInstanceException(DestoryInstanceStatus.FailedInvalidIdentifer, "instanceIdentifier cannot be null or empty.");
            }

            VoiceInstanceHost instance = null;

            if (!_voiceHostInstances.TryRemove(instanceIdentifier, out instance))
            {
                throw new DestoryInstanceException(DestoryInstanceStatus.FailedNotFound, "Cannot destroy instance. The instance identifier specified could not be found, or removed.");
            }

            try
            {
                instance.Dispose();
            }
            catch (Exception ex)
            {
                throw new DestoryInstanceException(DestoryInstanceStatus.FailedHostError, "Cannot close the service host: " + ex.Message, ex);
            }

            try
            {
                if (CallbackClient != null)
                {
                    CallbackClient.ReceiveInstanceDestroyed(instanceIdentifier);
                }
            }
            catch (Exception ex)
            {
                Debug.WriteLine("DestroyInstance failed to notify central service: " + ex.Message);
            }
        }

        public void UpdateCallbackClient(IVoiceHostCallbackClient callbackClient)
        {
            if (callbackClient == null)
            {
                return;
            }

            if (CallbackClient == null || (CallbackClient as ICommunicationObject).State != CommunicationState.Opened)
            {
                CallbackClient = callbackClient;
            }

        }

        public void StartNewInstance(IVoiceHostCallbackClient callbackClient, string instanceIdentifier, int ownerUserID)
        {
            if (!_isInitialized)
            {
                throw new Exception("VoiceInstanceManager has not been initialized!");
            }

            UpdateCallbackClient(callbackClient);

            if (string.IsNullOrEmpty(instanceIdentifier))
            {
                throw new ProvisionInstanceException(ProvisionInstanceStatus.FailedInvalidIdentifer, "instanceIdentifier cannot be null or empty.");
            }

            if (ownerUserID < 1)
            {
                throw new ProvisionInstanceException(ProvisionInstanceStatus.FailedInvalidUserID, "ownerUserID must be a valid number.");
            }

            if (_voiceHostInstances.ContainsKey(instanceIdentifier))
            {
                throw new ProvisionInstanceException(ProvisionInstanceStatus.FailedAlreadyExists, "Cannot provision instance. The instance identifier specified already exists: " + instanceIdentifier);
            }


            var voiceInstanceHost = new VoiceInstanceHost(instanceIdentifier, ownerUserID, _portNumber);
            if (!_voiceHostInstances.TryAdd(instanceIdentifier, voiceInstanceHost))
            {
                throw new ProvisionInstanceException(ProvisionInstanceStatus.FailedAlreadyExists, "Cannot complete instance provisioning. It could not be added to the central dictionary.");
            }

        }

        public void Dispose()
        {
            _isAlive = false;

            foreach (var instance in _voiceHostInstances.Values)
            {
                try
                {
                    instance.Dispose();
                }
                catch (Exception ex)
                {
                    Console.WriteLine("Failed to close voice host: " + ex.Message);
                }

            }
        }
    }
}
