﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.ServiceModel;
using System.Threading.Tasks;
using System.Timers;
using Curse.Logging;
using Curse.SocketInterface;
using Curse.SocketMessages;
using Curse.SocketServer;
using Curse.Voice.Contracts;
using Curse.Voice.Helpers;
using Curse.Voice.HostManagement;
using Curse.Voice.HostManagement.Exceptions;
using Curse.Voice.HostManagement.Responses;
using System.Threading;
using Curse.CloudServices;
using Curse.Friends.ServiceClients;
using Curse.Friends.Statistics;
using Curse.Voice.HostManagement.Models;
using Curse.Voice.HostRuntime.VoiceService;
using Curse.WebRTC;

namespace Curse.Voice.HostRuntime
{
    
    public class VoiceServer : SocketServer.SocketServer
    {
        private readonly LogCategory Logger = new LogCategory("VoiceServer") { AlphaLevel = LogLevel.Trace };
        private bool _isAlive = true;
        private readonly ConcurrentDictionary<string, VoiceInstance> _voiceHostInstances = new ConcurrentDictionary<string, VoiceInstance>();        
        public static readonly VoiceServer Instance = new VoiceServer(HostRuntimeConfiguration.Instance.MaxConnections, HostRuntimeConfiguration.Instance.VoicePortNumbers.Length);
        public IVoiceHostCallbackClient CallbackClient { get; private set; }
        private bool _isInitialized = false;
        private Version _minimumClientVersion = new Version(6, 0, 0, 0); // For testing
                
        private int _maxUsersPerVoiceSession = 0;
        private int _maxUsersPerVideoSession = 0;
        private string _externalID;
        
        private const int MaxDestroyQueue = 10000;
        private readonly ConcurrentQueue<string> _destroyQueue = new ConcurrentQueue<string>();

        public readonly ConcurrentQueue<Tuple<string, int>> MaxUsersQueue = new ConcurrentQueue<Tuple<string, int>>();

        public int TotalInstanceCount
        {
            get { return _voiceHostInstances.Count; }
        }

        public int ActiveInstanceCount
        {
            get { return _voiceHostInstances.Count(p => p.Value.UserCount > 1); }
        }

        public int ShutdownInstanceCount
        {
            get { return _voiceHostInstances.Count(p => p.Value.UserCount > 1 && !p.Value.IsFailingOver); }
        }

        public int ConnectionCount
        {
            get { return _numConnectedSockets + RTCPeerConnection.Count; }
        }

        private readonly System.Timers.Timer _transmitDestroyTimer;

        private readonly RTCRelayServer _relayServer = new RTCRelayServer();
        private readonly SocketUdpPool _udpSendPool;

        public RTCRelayServer RelayServer { get { return _relayServer; } }

        private VoiceServer(int maxConnections, int numberOfPorts)
            : base(maxConnections, numberOfPorts, false)
        {            
            ContractDispatcher.AddContractDispatcher<JoinSessionRequest>(OnJoinSession);
            ContractDispatcher.AddContractDispatcher<GetUsersRequest>(OnGetUsers);
            ContractDispatcher.AddContractDispatcher<UpdateUserRequest>(OnUpdateUser);
            ContractDispatcher.AddContractDispatcher<TransmitStartRequest>(OnStartTransmit);
            ContractDispatcher.AddContractDispatcher<TransmitEndRequest>(OnEndTransmit);
            ContractDispatcher.AddContractDispatcher<ChatMessageRequest>(OnChatMessageRequest);
            ContractDispatcher.AddContractDispatcher<LeaveSessionRequest>(OnLeaveSession);
            ContractDispatcher.AddContractDispatcher<SessionDescriptionAnswer>(OnSdpAnswer);
            ContractDispatcher.AddContractDispatcher<ReJoinSessionRequest>(OnReJoinSession);

            // Moderation requests
            ContractDispatcher.AddContractDispatcher<ModKickUserRequest>(OnKickUserRequest);
            ContractDispatcher.AddContractDispatcher<ModMuteUserRequest>(OnMuteUserRequest);
            ContractDispatcher.AddContractDispatcher<ModDeafenUserRequest>(OnDeafenUserRequest);

            new Thread(DisconnectIdleInstancesThread) { IsBackground = true }.Start();
            new Thread(UpdateMaxUsersThread) { IsBackground = true }.Start();
            new Thread(UpdateConfigurationThread) { IsBackground = true }.Start();

            _transmitDestroyTimer = new System.Timers.Timer(1000);
            _transmitDestroyTimer.Elapsed += TransmitDestroyQueueThread;
            _transmitDestroyTimer.AutoReset = true;
            _transmitDestroyTimer.Enabled = true;
            
            GroupCallbackManager.Initialize();
            VoiceInstanceScheduler.Start();

            _udpSendPool = new SocketUdpPool(_bufferManager, _maxConnections, RTCChannelProcessor.Completed);

        }

        public RTCChannelProcessor CreateUdpProcessor(ISocketInterface socketInterface, RTCSocketPair sockets)
        {
            return new RTCChannelProcessor(socketInterface, _udpSendPool, _udpSendPool.Get(), sockets);
        }

        public void Initialize(string externalID, int maxUsersPerVoiceSession, int maxUsersPerVideoSession)
        {
            VoiceServerLog.Info("Voice Server Initializing");
            SetInitialConfiguration(externalID, maxUsersPerVoiceSession, maxUsersPerVideoSession);
            _isInitialized = true;
            VoiceServerLog.Info("Voice Server Initialized");
        }

        public void SetInitialConfiguration(string externalID, int maxUsersPerVoiceSession, int maxUsersPerVideoSession)
        {
            _externalID = externalID;
            UpdateConfiguration(maxUsersPerVoiceSession, maxUsersPerVideoSession);            
            Logger.Info("Initial voice server configuration set", new { maxUsersPerVoiceSession, maxUsersPerVideoSession });
        }

        public void UpdateConfiguration(int maxUsersPerVoiceSession, int maxUsersPerVideoSession)
        {
            var hasChanges = false;
            if (_maxUsersPerVoiceSession != maxUsersPerVoiceSession)
            {
                hasChanges = true;
                _maxUsersPerVoiceSession = maxUsersPerVoiceSession;    
            }

            if (_maxUsersPerVideoSession != maxUsersPerVideoSession)
            {
                hasChanges = true;
                _maxUsersPerVideoSession = maxUsersPerVideoSession;
            }

            if (hasChanges)
            {
                Logger.Info("Voice server configuration updated", new { maxUsersPerVoiceSession, maxUsersPerVideoSession });    
            }            
        }

        public override void Start(IPEndPoint[] localEndPoints)
        {
            base.Start(localEndPoints);

            _relayServer.Start(_bufferManager, localEndPoints, _maxConnections, LookupClient);
        }

        public void UpdateMinimumClientVersion(string versionString)
        {
            Version minimumClientVersion;
            if (!Version.TryParse(versionString, out minimumClientVersion))
            {
                return;
            }
            _minimumClientVersion = minimumClientVersion;
        }

        public void BeginShutdown()
        {
            State = SocketServerState.Stopping;
        }

        public void Failover(FailoverInstructions[] failovers)
        {
            State = SocketServerState.FailingOver;

            foreach (var failover in failovers)
            {
                Failover(failover);
            }

            // Make sure our logs are submitted
            VoiceServerLog.Shutdown();
        }

        public void Failover(FailoverInstructions failover)
        {
            if (failover.Status != FailoverStatus.Successful)
            {
                return;
            }

            VoiceInstance instance;
            if (!_voiceHostInstances.TryGetValue(failover.InstanceCode, out instance) || instance == null)
            {
                VoiceServerLog.Info("Failover: Failed to find instance '" + failover.InstanceCode + "' scheduled for failover.");
                return;
            }

            if (instance.UserCount == 0)
            {
                Logger.Warn("Failover: Instance '" + failover.InstanceCode + "' will be skipped. It has no connected users.");
                return;
            }

            try
            {
                // Flag this instance as being failed over and reset the cleanup timer
                instance.IsFailingOver = true;
                instance.DateLastAccessed = DateTime.UtcNow;

                Logger.Debug("Failover: Starting failover for reason: " + failover.Reason, new {instance.Identifier, instance.Type, instance.Mode});

                var currentAttempt = 1;
                var startTime = DateTime.UtcNow;
                FailoverNotificationReason reason;

                switch (failover.Reason)
                {
                    case FailoverReason.ChangingMode:
                        reason = FailoverNotificationReason.ChangingMode;
                        break;

                    default:
                    case FailoverReason.HostUpdating:
                        reason = FailoverNotificationReason.HostUpdating;
                        break;
                }

                while (instance.UserCount > 0)
                {
                    instance.BroadcastFailover(failover.HostName, failover.IPAddress, failover.Port, reason);
                    Thread.Sleep(currentAttempt * 100);
                    if (++currentAttempt > 5)
                    {
                        break;
                    }
                }

                var totalAttempts = currentAttempt - 1;
                var duration = DateTime.UtcNow - startTime;

                if (instance.UserCount > 0)
                {
                    Logger.Warn("Failover: Instance '" + failover.InstanceCode + "' could not be gracefully failed over.", new { ConnectedUsers = instance.UserCount, Attempts = totalAttempts, DurationMilliseconds = duration.TotalMilliseconds });
                }
                else
                {
                    Logger.Debug("Failover: Instance '" + failover.InstanceCode + "' has been gracefully failed over.", new {Attempts = totalAttempts, DurationMilliseconds = duration.TotalMilliseconds});
                }
            }
            catch (Exception ex)
            {
                VoiceServerLog.Exception(ex, "Failover: Instance failover of '" + failover.InstanceCode + "' generated an error.");
            }
        }

        #region Contract Handlers

        private void OnStartTransmit(ISocketInterface socketInterface, TransmitStartRequest request)
        {
            var session = (VoiceSession)socketInterface.Session;
            if (session == null)
            {
                VoiceServerLog.Warn("TransmitStartRequest received before Session was set");
                return;
            }

            if (!session.CanSpeak)
            {
                // Don't do anything if they can't speak
                return;
            }

            if (session.UseServerMute)
            {
                session.IsSelfMuted = false;
            }
            session.Parent.BroadcastStartTransmit(session.VoiceSessionMember);
        }

        private void OnEndTransmit(ISocketInterface socketInterface, TransmitEndRequest request)
        {
            var session = (VoiceSession)socketInterface.Session;
            if (session == null)
            {
                VoiceServerLog.Warn("TransmitEndRequest received before Session was set");
                return;
            }

            if (session.UseServerMute)
            {
                session.IsSelfMuted = true;
            }
            session.Parent.BroadcastEndTransmit(session.VoiceSessionMember);
        }

        private void OnUpdateUser(ISocketInterface socketInterface, UpdateUserRequest updateUserRequest)
        {
            var session = (VoiceSession)socketInterface.Session;
            if (session == null || session.VoiceSessionMember == null)
            {
                Logger.Warn("Attempt to transmit without a session or user.");
                return;
            }

            var shouldBroadcast = false;

            if (updateUserRequest.DisplayName != null)
            {
                if (session.Parent.Users.Any(p => p.DisplayName.Equals(updateUserRequest.DisplayName.Trim(), StringComparison.InvariantCultureIgnoreCase) && p.ClientID != session.VoiceSessionMember.ClientID))
                {
                    VoiceServerLog.Debug("OnUpdateUser - Duplicate name '" + updateUserRequest.DisplayName + "'");
                }
                else
                {
                    session.VoiceSessionMember.DisplayName = updateUserRequest.DisplayName;
                    shouldBroadcast = true;
                }

            }

            if (updateUserRequest.AvatarUrl != null)
            {
                session.VoiceSessionMember.AvatarUrl = updateUserRequest.AvatarUrl;
                shouldBroadcast = true;
            }

            if (updateUserRequest.InGameName != null)
            {
                session.VoiceSessionMember.InGameName = updateUserRequest.InGameName;
                shouldBroadcast = true;
            }

            if (updateUserRequest.InGameRegion != null)
            {
                session.VoiceSessionMember.InGameRegion = updateUserRequest.InGameRegion;
                shouldBroadcast = true;
            }

            if (updateUserRequest.SupportsVideo.HasValue)
            {
                session.SupportsVideo = updateUserRequest.SupportsVideo.Value;
                // Don't broadcast changes to this property
            }

            if (updateUserRequest.VideoCodec != null)
            {
                // Use a codec with no name to turn off video since a null object means no update
                session.VoiceSessionMember.VideoCodec = updateUserRequest.VideoCodec.Name != null ? updateUserRequest.VideoCodec : null;
                shouldBroadcast = true;
            }

            // Notify other users, but only if something other than SupportsVideo changed since we shouldn't leak this info
            if (shouldBroadcast)
            {
                session.Parent.BroadcastUserUpdate(session.VoiceSessionMember);
            }
        }

        private void OnChatMessageRequest(ISocketInterface socketInterface, ChatMessageRequest chatMessageRequest)
        {
            var session = (VoiceSession)socketInterface.Session;
            if (session == null || session.VoiceSessionMember == null)
            {
                Logger.Warn("Attempt to transmit without a session or user.");
                return;
            }

            var response = new ChatMessageResponse
            {
                Timestamp = DateTime.UtcNow,
                ClientID = chatMessageRequest.ClientID,
            };

            try
            {
                FriendsStatsManager.Current.GroupMessagesSent.Track(session.Parent.UniqueID);
                FriendsStatsManager.Current.GroupMessagesSentByPlatform.Track(0); // Hardcode to Windows, for now.

                if (string.IsNullOrEmpty(chatMessageRequest.Body) || chatMessageRequest.Body.Length > 2000)
                {
                    response.Status = ChatMessageStatus.MessageTooLarge;
                    return;
                }

                if (session.IsFlooding())
                {
                    response.Status = ChatMessageStatus.FloodControl;
                    return;
                }

                response.Status = ChatMessageStatus.Successful;
                session.Parent.BroadcastChatMessage(session, chatMessageRequest.Body);
            }
            catch (Exception ex)
            {
                VoiceServerLog.Exception(ex, "Unhandled exception while receiving a chat message.");
                response.Status = ChatMessageStatus.Error;
            }
            finally
            {
                socketInterface.SendContract(response);
            }


        }

        private void OnGetUsers(ISocketInterface socketInterface, GetUsersRequest getUsersContract)
        {
            try
            {
                var session = (VoiceSession)socketInterface.Session;
                if (session == null)
                {
                    VoiceServerLog.Warn("GetUserRequest received before Session was set");
                    return;
                }

                var resp = new GetUsersResponse()
                {
                    Users = session.Parent.Users,
                    SessionType = session.Parent.Type,
                    PendingUsers = session.Parent.PendingUsers,
                    Timestamp = DateTime.UtcNow
                };
                socketInterface.SendContract(resp);

                if (session.UseSdp)
                {
                    session.SendOffer();
                }
            }
            catch (Exception ex)
            {

                Logger.Error(ex, "OnGetUsers");
            }

        }

        private int GetMaxUsersPerSession(VoiceInstanceMode mode)
        {
            switch (mode)
            {
                case VoiceInstanceMode.Video:
                    return _maxUsersPerVideoSession;

                case VoiceInstanceMode.Audio:
                default:
                    return _maxUsersPerVoiceSession;
            }
        }

        private void OnJoinSession(ISocketInterface socketInterface, JoinSessionRequest joinSessionRequest)
        {

            var resp = new JoinSessionResponse()
            {
                Status = JoinSessionStatus.FailedUnhandledException,
                Timestamp = DateTime.UtcNow
            };

            VoiceServerLog.Debug("OnJoinSession - Received a join request for session '" + joinSessionRequest.SessionID + "' from user '" + joinSessionRequest.DisplayName + "' and user id '" + (joinSessionRequest.UserID.HasValue ? joinSessionRequest.UserID.Value.ToString() : "Unknown") + "'");

            var session = default(VoiceSession);
            try
            {
                Version version = null;

                var userID = joinSessionRequest.UserID;

                // Check their client version
                if (!Version.TryParse(joinSessionRequest.ClientVersion, out version) || version < _minimumClientVersion)
                {
                    resp.Status = JoinSessionStatus.InvalidClientVersion;
                    socketInterface.SendContract(resp);
                    return;
                }

                // Validate the display name
                if (joinSessionRequest.DisplayName == null || joinSessionRequest.DisplayName.Trim().Length < 1)
                {
                    resp.Status = JoinSessionStatus.FailedNameAlreadyTaken;
                    socketInterface.SendContract(resp);
                    return;
                }

                VoiceInstance instance = null;

                // Look for the voice instance
                if (!_voiceHostInstances.TryGetValue(joinSessionRequest.SessionID, out instance))
                {
                    VoiceServerLog.Debug("OnJoinSession - Session not found: " + joinSessionRequest.SessionID);
                    resp.Status = JoinSessionStatus.SessionNotFound;
                    return;
                }

                if (instance.IsExpired || instance.IsFailingOver)
                {
                    VoiceServerLog.Debug("OnJoinSession - Session expired or failed over.");
                    resp.Status = JoinSessionStatus.SessionNotFound;
                    return;
                }
                
                // Ensure the session is not full
                if (instance.UserCount >= GetMaxUsersPerSession(instance.Mode))
                {
                    resp = new JoinSessionResponse() { Status = JoinSessionStatus.FailedSessionFull };
                    return;
                }

                // Check access token as needed
                if (instance.RequiresAccessToken)
                {
                    // Ensure access token is present
                    if (joinSessionRequest.AuthToken == null)
                    {
                        VoiceServerLog.Warn("OnJoinSession - Authentication token is missing.", new { joinSessionRequest.ClientVersion, joinSessionRequest.SessionID });
                        resp.Status = JoinSessionStatus.Forbidden;
                        return;
                    }

                    if (!joinSessionRequest.AccessToken.HasValue)
                    {
                        VoiceServerLog.Warn("OnJoinSession - Access token is missing.", new { joinSessionRequest.ClientVersion, joinSessionRequest.SessionID });
                        resp.Status = JoinSessionStatus.Forbidden;
                        return;
                    }

                    var token = AuthenticationTokenData.FromString(joinSessionRequest.AuthToken);

                    if (!token.IsValid)
                    {
                        VoiceServerLog.Warn("OnJoinSession - Authentication token is invalid.", new { joinSessionRequest.ClientVersion, joinSessionRequest.SessionID });
                        resp.Status = JoinSessionStatus.Forbidden;
                        return;
                    }

                    if (token.UserID == 0)
                    {
                        VoiceServerLog.Warn("OnJoinSession - Authentication token is anonymous.", new { joinSessionRequest.ClientVersion, joinSessionRequest.SessionID });
                        resp.Status = JoinSessionStatus.Forbidden;
                        return;
                    }

                    // Get the authenticated user ID.
                    userID = token.UserID;

                    // Test access token
                    if (!instance.CheckAccessToken(token.UserID, joinSessionRequest.AccessToken.Value))
                    {
                        resp.Status = JoinSessionStatus.Forbidden;
                        return;
                    }

                    // Remove any existing sessions for the authenticated user
                    var sessionMember = instance.GetSessionByUserID(token.UserID);
                    if (sessionMember != null)
                    {
                        instance.DisconnectUser(sessionMember.ID, sessionMember.ID, UserDisconnectReason.Duplicate);
                    }
                }
                else // Prevent duplicate names for insecure sessions
                {
                    if (instance.Users.Any(p => p.DisplayName.Equals(joinSessionRequest.DisplayName.Trim(), StringComparison.InvariantCultureIgnoreCase) && p.UserID != userID))
                    {
                        VoiceServerLog.Warn("User attempted to join a session with the same display name as another user '" + joinSessionRequest.DisplayName + "'", new { joinSessionRequest.UserID, joinSessionRequest.DisplayName });
                        resp.Status = JoinSessionStatus.FailedNameAlreadyTaken;
                        return;
                    }
                }

                VoiceUserPermissions permissions = null;
                bool? isModMuted = null;
                bool? isModDeafened = null;

                switch (instance.Type)
                {
                    case VoiceInstanceType.AdHoc:
                    case VoiceInstanceType.MultiFriend:
                        permissions = !userID.HasValue || instance.CreatorUserID != userID.Value
                            ? new VoiceUserPermissions {CanSpeak = true}
                            : new VoiceUserPermissions
                            {
                                CanSpeak = true,
                                CanModMute = true,
                                CanModDeafen = true,
                                CanModKick = true
                            };
                        break;
                    case VoiceInstanceType.Group:
                        if (instance.GroupID.HasValue && userID.HasValue)
                        {
                    var modResponse = FriendsServiceClients.Instance.CallsAdmin.GetModerationStatus(instance.GroupID.Value, userID.Value);
                            if (modResponse.Success)
                            {
                                Logger.Trace("Successfully retrieved group user's moderation status.", new {userID, modResponse});
                                isModMuted = modResponse.Response.IsMuted;
                                isModDeafened = modResponse.Response.IsDeafened;
                                permissions = new VoiceUserPermissions
                                {
                                    UserID = userID.Value,
                                    BestRoleRank = modResponse.Response.Permissions.BestRoleRank,
                                    CanModMute = modResponse.Response.Permissions.CanModMute,
                                    CanSpeak = modResponse.Response.Permissions.CanSpeak,
                                    CanModDeafen = modResponse.Response.Permissions.CanModDeafen,
                                    CanModKick = modResponse.Response.Permissions.CanModKick
                                };
                            }
                            else
                            {
                                Logger.Warn("Unable to determine moderation status (permissions, muted, deafened). Assuming defaults.");
                                permissions = new VoiceUserPermissions {CanSpeak = true};
                            }
                        }
                        else
                        {
                            Logger.Warn("Group ID or User ID was null on joining session", new {instance.GroupID, userID});
                            permissions = new VoiceUserPermissions {CanSpeak = true};
                        }
                        break;
                    default:
                        permissions = new VoiceUserPermissions { CanSpeak = true };
                        break;
                }

                session = instance.CreateSession(userID, joinSessionRequest, socketInterface, isModMuted, isModDeafened, permissions);
                socketInterface.Session = session;
                socketInterface.IsAuthenticated = true;

                resp = new JoinSessionResponse()
                {
                    ClientID = socketInterface.ClientID,
                    Status = JoinSessionStatus.Successful,
                    Users = instance.Users,
                    CurrentUserID = session.ID,
                    Type = instance.Type,
                    PendingUsers = instance.PendingUsers,
                    Timestamp = DateTime.UtcNow
                };
            }
            catch (Exception ex)
            {
                VoiceServerLog.Exception(ex, "OnJoinSession - Unhandled Exception!");
            }
            finally
            {
                socketInterface.SendContract(resp);
                if (resp.Status != JoinSessionStatus.Successful)
                {
                    socketInterface.RaiseDisconnect(SocketDisconnectReason.HandshakeFailure);
                }
                else if (joinSessionRequest.WebRTCType != MultiStreamType.None && session != null)
                {
                    session.SendOffer();
                }
            }
        }

        private void OnLeaveSession(ISocketInterface socketInterface, LeaveSessionRequest request)
        {
            var session = (VoiceSession)socketInterface.Session;
            if (session == null || session.VoiceSessionMember == null)
            {
                Logger.Warn("Attempt to leave without a session or user.");
                return;
            }

            var instance = session.Parent;
            if (instance == null)
            {
                Logger.Warn("Attempt to leave without a session parent.");
                return;
            }

            try
            {

                instance.UserLeft(session);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "OnLeaveSession");
            }
        }

        private void OnSdpAnswer(ISocketInterface socketInterface, SessionDescriptionAnswer answer)
        {
            try
            {
                var session = (VoiceSession)socketInterface.Session;
                if (session == null)
                {
                    VoiceServerLog.Warn("SessionDescriptionAnswer received before Session was set");
                    return;
                }

                session.SetAnswer(answer);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "OnSdpAnswer");
            }
        }

        private void OnReJoinSession(ISocketInterface socketInterface, ReJoinSessionRequest reJoin)
        {
            try
            {
                var session = (VoiceSession)socketInterface.Session;
                if (session == null)
                {
                    Logger.Warn("Attempt to re-join without a session");
                    return;
                }

                var instance = session.Parent;
                if (instance == null)
                {
                    Logger.Warn("Attempt to re-join without a session parent");
                    return;
                }

                instance.ReJoin(session);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "OnReJoinSession");
            }
        }

        #region Moderation

        private void OnDeafenUserRequest(ISocketInterface socketInterface, ModDeafenUserRequest request)
        {
            var response = new ModDeafenUserResponse
            {
                Success = false
            };
            try
            {
                var session = (VoiceSession) socketInterface.Session;
                if (session == null)
                {
                    VoiceServerLog.Warn("OnDeafenUserRequest received before Session was set");
                    return;
                }

                var instance = session.Parent;
                if (instance == null)
                {
                    Logger.Warn("Attempt to deafen user without a session parent.");
                    return;
                }

                response.Success = instance.ModDeafenUser(session, request.UserID, request.Deafen);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "OnDeafenUserRequest");
            }
            finally
            {
                socketInterface.SendContract(response);
            }
        }

        private void OnMuteUserRequest(ISocketInterface socketInterface, ModMuteUserRequest request)
        {
            var response = new ModMuteUserResponse
            {
                Success = false
            };
            try
            {
                var session = (VoiceSession)socketInterface.Session;
                if (session == null)
                {
                    VoiceServerLog.Warn("OnMuteUserRequest received before Session was set");
                    return;
                }

                var instance = session.Parent;
                if (instance == null)
                {
                    Logger.Warn("Attempt to mute user without a session parent.");
                    return;
                }

                response.Success = instance.ModMuteUser(session, request.UserID, request.Mute);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "OnMuteUserRequest");
            }
            finally
            {
                socketInterface.SendContract(response);
            }
        }

        private void OnKickUserRequest(ISocketInterface socketInterface, ModKickUserRequest request)
        {
            var response = new ModKickUserResponse
            {
                Success = false
            };
            try
            {
                var session = (VoiceSession)socketInterface.Session;
                if (session == null)
                {
                    VoiceServerLog.Warn("OnKickUserRequest received before Session was set");
                    return;
                }

                var instance = session.Parent;
                if (instance == null)
                {
                    Logger.Warn("Attempt to kick user without a session parent.");
                    return;
                }

                response.Success = instance.ModKickUser(session, instance.GetSessionByUserID(request.UserID));
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "OnKickUserRequest");
            }
            finally
            {
                socketInterface.SendContract(response);
            }
        }

        #endregion


        #endregion

        #region Management

        private void DisconnectIdleInstancesThread()
        {
            while (_isAlive)
            {
                Thread.Sleep(TimeSpan.FromSeconds(10));

                try
                {
                    DisconnectIdleInstances();
                }
                catch (Exception ex)
                {
                    VoiceServerLog.Exception(ex, "Failed to disconnect idle instances.");
                }
            }
        }

        private void UpdateMaxUsersThread()
        {
            
            while (_isAlive)
            {
                
                Thread.Sleep(TimeSpan.FromSeconds(1));

                try
                {
                    if (MaxUsersQueue.Count == 0)
                    {
                        continue;
                    }

                    var instanceList = new List<Tuple<string, int>>();                    
                    while (MaxUsersQueue.Count > 0 && instanceList.Count < 100) // Only transmit batched of 100 every second
                    {
                        Tuple<string, int> tuple;
                        if (MaxUsersQueue.TryDequeue(out tuple))
                        {
                            instanceList.Add(tuple);                            
                        }                        
                    }

                    if (instanceList.Any())
                    {
                        TryUpdateMaxUsersToService(5, instanceList);
                    }                    
                }
                catch (Exception ex)
                {
                    VoiceServerLog.Exception(ex, "General exception in UpdateMaxUsersThread");
                }                
            }
        }

        private void TryUpdateMaxUsersToService(int maxAttempts, List<Tuple<string, int>> instanceToUserList)
        {
            var attempt = 0;

            while (true)
            {
                try
                {
                    UpdateMaxUsersToService(instanceToUserList);
                    return;
                }
                catch (Exception ex)
                {
                    var logMessage = "Failed to send voice instance max user numbers to the central service after " + attempt + " attempts";
                    
                    if (attempt++ < maxAttempts)
                    {
                        Logger.Warn(ex, logMessage);
                        Thread.Sleep(1000);
                    }
                    else
                    {
                        Logger.Error(ex, logMessage);
                        return;
                    }
                }
            }
            
        }
        
        private void UpdateMaxUsersToService(List<Tuple<string, int>> instanceToUserList)
        {           
            var groupedInstances = instanceToUserList.GroupBy(p => p.Item1).Select(p1 => new UpdateInstanceUserCountRequest{InstanceCode = p1.Key, MaxUsers = p1.Max(p2=>p2.Item2)}).ToArray();
            using (var client = new CurseVoiceServiceClient(BindingHelper.GetBasicHttpBinding(30), new EndpointAddress(HostRuntimeConfiguration.Instance.CentralServiceUrl)))
            {   
                if (!client.UpdateInstancesWithMaxUsers(HostRuntimeConfiguration.Instance.CentralServiceApiKey, groupedInstances))
                {
                    Logger.Warn("Failed to update max users.", new {groupedInstances});
                }
            }
            
        }


        private volatile bool _isTransmittingQueue = false;
        private void TransmitDestroyQueueThread(object t, ElapsedEventArgs e)
        {
            if (_isTransmittingQueue)
            {
                Logger.Warn("Skipping transmit destroy queue. It is already running.");
                return;
            }

            _isTransmittingQueue = true;

            try
            {
                try
                {
                    var task = Task.Factory.StartNew(TransmitDestroyQueue);

                    if (!task.Wait(10000))
                    {
                        VoiceServerLog.Warn("Failed to finish transmitting the destroy queue after 10 seconds!");
                    }
                }
                catch (Exception ex)
                {
                    VoiceServerLog.Exception(ex, "Failed to transmit destroyed instance queue.");
                }

                try
                {
                    CallbackHealthCheck();
                }
                catch (Exception ex)
                {
                    VoiceServerLog.Exception(ex, "Failed to perform callback health check.");
                }
            }            
            finally
            {
                _isTransmittingQueue = false;
            }            
        }

        private void CallbackHealthCheck()
        {
            if (!ShouldCallbackAlert())
            {
                return;
            }

            VoiceServerLog.Error("Idle Instance Cleanup: The server has been unable to reach the central service with idle instance cleanup instructions for more than 15 minutes.");
        }

        private bool ShouldCallbackAlert()
        {
            // Insufficient data to make a determination
            if (!_lastCallbackFailure.HasValue || !_lastCallbackSuccess.HasValue)
            {
                return false;
            }

            // We've had a success since our last failure
            if (_lastCallbackSuccess.Value > _lastCallbackFailure.Value)
            {
                return false;
            }

            // Our last failure was in the last 10 minutes, and we have not had a success in 15
            if (DateTime.UtcNow.Subtract(_lastCallbackFailure.Value) < TimeSpan.FromMinutes(10)
                && DateTime.UtcNow.Subtract(_lastCallbackSuccess.Value) > TimeSpan.FromMinutes(15))
            {
                return true;
            }

            return false;
        }

        private bool IsCallbackClientOpen
        {
            get
            {
                var client = CallbackClient as ICommunicationObject;
                return client != null && client.State == CommunicationState.Opened;
            }

        }

        private DateTime? _lastCallbackSuccess = null;
        private DateTime? _lastCallbackFailure = null;

        private void TransmitDestroyQueue()
        {
            if (!_isAlive)
            {
                Logger.Info("The server is shutting down so the destroy queue will not be communicated to the central service.");
                return;
            }

            if (_destroyQueue.Count == 0)
            {
                return;
            }

            if (!IsCallbackClientOpen)
            {
                Logger.Warn("The central service callback client is null or not open. Destroyed instances will not be communicated back to the central service, and will continue to queue.");
                return;
            }

            var destroyList = new List<string>();
            string instanceIdentifier;
            while (_destroyQueue.TryDequeue(out instanceIdentifier) && destroyList.Count < 100)
            {
                destroyList.Add(instanceIdentifier);
            }

            foreach (var id in destroyList)
            {
                if (!_isAlive)
                {
                    break;
                }
                try
                {
                    CallbackClient.ReceiveInstanceDestroyed(id);
                    _lastCallbackSuccess = DateTime.UtcNow;
                }
                catch
                {
                    _lastCallbackFailure = DateTime.UtcNow;
                    _destroyQueue.Enqueue(id);
                }
            }
        }

        private void DisconnectIdleInstances()
        {

            var expiredInstances = _voiceHostInstances.Values.Where(p => p.IsExpired).ToArray();

            foreach (var instance in expiredInstances)
            {
                if (!_isAlive)
                {
                    break;
                }

                try
                {
                    DestroyInstance(instance.Identifier);
                }
                catch (DestroyInstanceException ex)
                {
                    VoiceServerLog.Warn("Idle Instance Cleanup - Destroy Instance DestoryInstanceException: " + ex.Message);
                }
                catch (Exception ex)
                {
                    VoiceServerLog.Exception(ex, "Idle Instance Cleanup - Destroy Instance");
                }
            }

        }

        public void DestroyInstance(string instanceIdentifier)
        {
            if (string.IsNullOrEmpty(instanceIdentifier))
            {
                throw new DestroyInstanceException(DestoryInstanceStatus.FailedInvalidIdentifer, "instanceIdentifier cannot be null or empty.");
            }

            VoiceInstance instance = null;

            if (!_voiceHostInstances.TryRemove(instanceIdentifier, out instance))
            {
                return;
            }

            // Don't report instances that were failed over (central service already knows)
            if (instance.IsFailingOver)
            {
                return;
            }

            instance.HandleEnded();
            QueueDestroyCall(instanceIdentifier);
        }

        public VoiceInstance GetInstance(string instanceIdentifier)
        {
            VoiceInstance instance = null;

            if (!_voiceHostInstances.TryGetValue(instanceIdentifier, out instance))
            {
                return null;
            }

            return instance;
        }

        private void QueueDestroyCall(string instanceIdentifier)
        {
            if (_destroyQueue.Count < MaxDestroyQueue)
            {
                _destroyQueue.Enqueue(instanceIdentifier);
            }
            else
            {
                VoiceServerLog.Error("Failed to transmit destroyed instance to central service, and local queue is full!");
            }
        }

        public void UpdateCallbackClient(OperationContext operationContext)
        {

            if (IsCallbackClientOpen)
            {
                return;
            }

            VoiceServerLog.Debug("Getting callback channel");
            var callbackChannel = operationContext.GetCallbackChannel<IVoiceHostCallbackClient>();

            if (callbackChannel == null || (callbackChannel as ICommunicationObject).State != CommunicationState.Opened)
            {
                return;
            }

            if (CallbackClient != null)
            {
                try
                {
                    (CallbackClient as ICommunicationObject).Abort();
                }
                catch (Exception ex)
                {
                    VoiceServerLog.Debug("Failed to abort prior callback client: " + ex.Message);
                }

            }

            CallbackClient = callbackChannel;
        }

        public void StartNewInstance(string instanceIdentifier, string joinCode, VoiceInstanceType type, VoiceInstanceMode mode, int? ownerUserID = null, Guid? groupID = null)
        {
            if (!_isInitialized)
            {
                throw new ProvisionInstanceException(ProvisionInstanceStatus.FailedHostError, "Cannot complete instance provisioning. The server is in the following state: '" + State + "'");
            }

            if (State != SocketServerState.Started)
            {
                throw new ProvisionInstanceException(ProvisionInstanceStatus.FailedHostError, "Cannot complete instance provisioning. The server is in the following state: '" + State + "'");
            }

            if (string.IsNullOrEmpty(instanceIdentifier))
            {
                throw new ProvisionInstanceException(ProvisionInstanceStatus.FailedInvalidIdentifer, "instanceIdentifier cannot be null or empty.");
            }

            if (ownerUserID.HasValue && ownerUserID.Value < 0)
            {
                throw new ProvisionInstanceException(ProvisionInstanceStatus.FailedInvalidUserID, "ownerUserID must be a valid number.");
            }

            if (groupID.HasValue && groupID == Guid.Empty)
            {
                throw new ProvisionInstanceException(ProvisionInstanceStatus.FailedInvalidUserID, "group ID must be a valid guid.");
            }

            if (_voiceHostInstances.ContainsKey(instanceIdentifier))
            {
                throw new ProvisionInstanceException(ProvisionInstanceStatus.FailedAlreadyExists, "Cannot provision instance. The instance identifier specified already exists: " + instanceIdentifier);
            }

            var voiceInstanceHost = new VoiceInstance(instanceIdentifier, joinCode, type, mode, ownerUserID, groupID);

            if (!_voiceHostInstances.TryAdd(instanceIdentifier, voiceInstanceHost))
            {
                throw new ProvisionInstanceException(ProvisionInstanceStatus.FailedAlreadyExists, "Cannot complete instance provisioning. It could not be added to the central dictionary.");
            }

            Task.Factory.StartNew(voiceInstanceHost.HandleCreated);
        }

        public InstanceStatistics[] GetInstanceStatistics()
        {
            var stats = new List<InstanceStatistics>();

            foreach (var instance in _voiceHostInstances.Values.ToArray())
            {
                stats.Add(new InstanceStatistics()
                {
                    Identifier = instance.Identifier,
                    DateCreated = instance.DateCreated,
                    UserCount = instance.UserCount,
                    InboundVoiceDataCounter = instance.InboundVoiceDataCounter,
                    OutboundVoiceDataCounter = instance.OutboundVoiceDataCounter,
                    InboundVideoDataCounter = instance.InboundVideoDataCounter,
                    OutboundVideoDataCounter = instance.OutboundVideoDataCounter,
                });
            }

            return stats.ToArray();
        }

        public ConnectionDiagnostics[] GetConnectionDiagnostics()
        {
            var list = new List<ConnectionDiagnostics>(_clients.Count);

            var utcNow = DateTime.UtcNow;
            var tickNow = Environment.TickCount;
            foreach (var client in _clients.Values.ToArray())
            {
                try
                {
                    var diagnostic = new ConnectionDiagnostics();
                    diagnostic.ClientID = client.ClientID;
                    diagnostic.DateLastHandshake = utcNow - TimeSpan.FromMilliseconds(tickNow - client.DateLastHandshake);
                    var addr = client.RemoteAddress;
                    if (addr != null)
                    {
                        diagnostic.RemoteEndPoint = addr.ToString();
                    }
                    else
                    {
                        diagnostic.RemoteEndPoint = "Dead Socket";
                    }

                    var session = client.Session as VoiceSession;
                    if (session != null)
                    {
                        diagnostic.SessionID = session.ID;
                    }
                    else
                    {
                        diagnostic.SessionID = -1;
                    }

                    list.Add(diagnostic);
                }
                catch (Exception ex)
                {
                    VoiceServerLog.Exception(ex, "Failed to create connection diagonistics!");
                }
            }

            foreach (var conn in RTCPeerConnection.All)
            {
                try
                {
                    var diagnostic = new ConnectionDiagnostics
                    {
                        ClientID = -1,
                        DateLastHandshake = DateTime.MinValue,
                        SessionID = -1,
                        RemoteEndPoint = conn.EndPoint != null ? conn.EndPoint.ToString() : "Dead Socket",
                    };

                    var session = conn.Owner as VoiceSession;
                    if (session != null)
                    {
                        diagnostic.SessionID = session.ID;

                        var client = session.ServerSocket;
                        if (client != null)
                        {
                            diagnostic.ClientID = client.ClientID;
                            diagnostic.DateLastHandshake = utcNow - TimeSpan.FromMilliseconds(tickNow - client.DateLastHandshake);
                        }
                    }

                    list.Add(diagnostic);
                }
                catch (Exception ex)
                {
                    VoiceServerLog.Exception(ex, "Failed to create RTC connection diagnostics!");
                }
            }

            return list.ToArray();

        }

        private void UpdateConfigurationThread()
        {
            var failCount = 0;
            while (_isAlive)
            {
                Thread.Sleep(TimeSpan.FromSeconds(30));

                try
                {
                    TryUpdateConfiguration();
                    failCount = 0;
                }
                catch (Exception ex)
                {
                    ++failCount;
                    if (failCount < 10)
                    {
                        Logger.Warn(ex, "Failed to update configuration");
                    }
                    else
                    {
                        // Only elevate to an error if this has failed for the past 5 minutes
                        Logger.Error(ex, "Failed to update configuration 10 times in a row");
                        failCount = 0;
                    }
                }
            }
        }

        private void TryUpdateConfiguration()
        {
            if (_externalID == null)
            {
                return;
            }

            using (var client = new CurseVoiceServiceClient(new BasicHttpBinding(), new EndpointAddress(HostRuntimeConfiguration.Instance.CentralServiceUrl)))
            {
                var resp = client.GetVoiceHostConfiguration(HostRuntimeConfiguration.Instance.CentralServiceApiKey, _externalID);

                if (resp == null || resp.Configuration == null)
                {
                    return;
                }
                
                UpdateConfiguration(resp.Configuration.MaxUsersPerVoiceSession, resp.Configuration.MaxUsersPerVideoSession);
            }
        }

        #endregion

        public override void Stop()
        {
            _isAlive = false;

            try
            {
                VoiceServerLog.Info("Shutting down...");

                if (_transmitDestroyTimer != null)
                {
                    _transmitDestroyTimer.Enabled = false;
                }

                _relayServer.ShutdownAll();
            }
            catch (Exception ex)
            {
                VoiceServerLog.Exception(ex);
            }
                        
            base.Stop();
            VoiceServerLog.Shutdown();
        }
    }
}
