﻿using System.Diagnostics;
using System.Globalization;
using System.Net;
using Curse.Logging;
using Curse.Voice.Contracts;
using System;
using Curse.SocketInterface;
using System.Security.Cryptography;
using Curse.Voice.HostManagement.Models;
using Curse.WebRTC;

namespace Curse.Voice.HostRuntime
{
    /// <summary>
    /// Represents one user session connected to a voice instance.
    /// </summary>
    public class VoiceSession : ISocketSession, IRTCStream
    {
        private static LogCategory Logger = new LogCategory("VoiceSession") { Throttle = TimeSpan.FromSeconds(30) };

        public RTCChannelProcessor UdpChannel { get; private set; }

        #region Constructor

        public VoiceSession(VoiceInstance parent, VoiceSessionMember voiceSessionMember, ISocketInterface serverSocket, MultiStreamType webRTCType)
        {            
            Parent = parent;
            VoiceSessionMember = voiceSessionMember;
            ServerSocket = serverSocket;            
            IsConnected = true;

            // Make sure PacketMilliseconds and SampleRate have non-zero values.
            // Fallback values are based on the current settings when these fields were added.
            if (VoiceSessionMember.CodecInfo == null)
            {
                VoiceSessionMember.CodecInfo = new CodecInfo
                {
                    Name = "Opus",
                    PacketMilliseconds = 40,
                    SampleRate = 48000,
                };
            }
            else
            {
                if (VoiceSessionMember.CodecInfo.PacketMilliseconds == 0)
                {
                    VoiceSessionMember.CodecInfo.PacketMilliseconds = 40;
                }
                if (VoiceSessionMember.CodecInfo.SampleRate == 0)
                {
                    VoiceSessionMember.CodecInfo.SampleRate = 48000;
                }
            }

            UdpChannel = VoiceServer.Instance.CreateUdpProcessor(serverSocket, null);

            if (webRTCType != MultiStreamType.None)
            {
                _rtcPeerConnection = new RTCPeerConnection(this, webRTCType, UdpChannel);

                if (UseServerMute)
                {
                    IsSelfMuted = true; // WebRTC connections start muted
                }
            }
        }

        #endregion

        #region Properties

        /// <summary>
        /// The provisioned voice instance that this session belongs to
        /// </summary>
        public VoiceInstance Parent { get; private set; }

        /// <summary>
        /// Information about the user that is associated with this session
        /// </summary>
        public VoiceSessionMember VoiceSessionMember { get; private set; }

        /// <summary>
        /// This is the user's local session identifier. It is not unique in a global scope.
        /// </summary> 
        public int ID { get { return VoiceSessionMember.ClientID; } }
        
        string ISocketSession.SessionID
        {
            get{ return ID.ToString(CultureInfo.InvariantCulture); }
        }

        public SymmetricAlgorithm EncryptionAlgorithm { get; private set; }

        public ISocketInterface ServerSocket { get; private set; }

        private DateTime? _lastChatMessageSent = null;
        
        public bool IsConnected { get; private set; }

        public bool ShouldReceiveNotifications
        {
            get { return IsConnected && !IsPendingDisconnect; }
        }

        public bool IsCreator
        {
            get { return VoiceSessionMember.UserID.HasValue && VoiceSessionMember.UserID.Value == Parent.CreatorUserID; }
        }

        public bool IsOwner
        {
            get { return ID == Parent.OwnerSessionID; }
        }
        
        public bool IsFlooding()
        {            
            if (!_lastChatMessageSent.HasValue)
            {
                _lastChatMessageSent = DateTime.UtcNow;
                return false;
            }

            if ((DateTime.UtcNow - _lastChatMessageSent.Value).TotalMilliseconds < 500)
            {
                return true;
            }

            _lastChatMessageSent = DateTime.UtcNow;
            return false;
        }

        public bool IsPendingDisconnect { get; private set; }

        public bool LoopbackEnabled { get; set; }

        public bool IsSelfMuted { get; set; }

        public bool IsModDeafened
        {
            get { return VoiceSessionMember.IsModDeafened; }
            set { VoiceSessionMember.IsModDeafened = value; }
        }

        public bool IsModMuted
        {
            get { return VoiceSessionMember.IsModMuted; }
            set { VoiceSessionMember.IsModMuted = value; }
        }

        public bool CanSpeak
        {
            get { return VoiceSessionMember.CanSpeak; }
            set { VoiceSessionMember.CanSpeak = value; }
        }

        public VoiceUserPermissions Permissions { get; set; }

        public bool SupportsVideo { get; set; }

        public bool HasVideo
        {
            get { return VoiceSessionMember.VideoCodec != null; }
        }

        #endregion

        /// <summary>
        /// Handles a socket disconnect
        /// </summary>
        void ISocketSession.Disconnect()
        {            
            IsConnected = false;
            IsPendingDisconnect = false;
            Parent.ClientDisconnected(this);

            if (UseWebRTC)
            {
                _rtcPeerConnection.Close();
            }

            UdpChannel.Dispose();
        }

        /// <summary>
        /// Sends a disconnect packet to the client, askng it to disconnect.
        /// If the client does not disconnect gracefully, the pending disconnect system will 
        /// </summary>
        /// <param name="initiatingUserID"></param>
        /// <param name="reason"></param>
        public void SendUserDisconnect(int initiatingUserID, UserDisconnectReason reason)
        {            
            NotifyUserDisconnect(new UserDisconnectNotification
            {
                InitiatingUserID = initiatingUserID,
                AffectedUserID = ID,
                Reason = reason,
                Timestamp = DateTime.UtcNow
            });

            IsPendingDisconnect = true;
            VoiceInstancePendingDisconnect.AddPendingDisconnect(Parent, this);
        }

        /// <summary>
        /// Forcefully disconnects the socket
        /// </summary>
        public void ForceDisconnect()
        {
            var serverSocket = ServerSocket;
            if (serverSocket == null)
            {
                return;
            }

            serverSocket.RaiseDisconnect(SocketDisconnectReason.UserInitiated);
        }

        public void Dispose()
        {
            ServerSocket = null;
            
            if (EncryptionAlgorithm != null)
            {
                try
                {
                    EncryptionAlgorithm.Dispose();
                }
                catch { }
            }
        }

        private void SendPacket(IContract packet)
        {
            if (!ShouldReceiveNotifications)
            {
                return;
            }

            var socket = ServerSocket as BaseSocketInterface;
            if (socket == null)
            {
                return;
            }
            socket.SendPacket(packet);
        }

        private void SendContract(IContract contract)
        {
            if (!ShouldReceiveNotifications)
            {
                return;
            }

            var socket = ServerSocket;
            if (socket == null)
            {
                return;
            }

            socket.SendContract(contract);
        }

        public void NotifyUserLeft(UserLeftNotification userLeftNotification)
        {
            SendContract(userLeftNotification);

            // Update the SDP when the user list changes
            if (UseSdp)
            {
                SendOffer();
            }
        }

        public void NotifyStartTransmit(TransmitStartNotification notification)
        {
            SendContract(notification);
        }

        public void NotifyEndTransmit(TransmitEndNotification notification)
        {
            SendContract(notification);
        }

        public void NotifyUserJoined(UserJoinedNotification userJoinedNotification)
        {
            SendContract(userJoinedNotification);

            // Update the SDP when the user list changes
            if (UseSdp)
            {
                SendOffer();
            }
        }

        public void NotifyUserUpdated(UserUpdatedNotification userUpdateNotification)
        {
            SendContract(userUpdateNotification);
        }

        public void NotifyFailover(FailoverNotification failoverNotification)
        {
            SendContract(failoverNotification);
        }

        public void NotifyChatMessage(ChatMessageNotification chatMessageNotification)
        {
            SendContract(chatMessageNotification);
        }

        public void NotifyUserDisconnect(UserDisconnectNotification notification)
        {
            SendContract(notification);
        }

        public void NotifyInstanceChanged(VoiceInstanceChangedNotification notification)
        {
            SendContract(notification);
        }

        public void NotifyPendingUser(AddPendingUsersNotification notification)
        {
            SendContract(notification);
        }

        public void NotifyRemovePendingUser(RemovePendingUserNotification notification)
        {
            SendContract(notification);
        }

        #region WebRTC

        private readonly RTCPeerConnection _rtcPeerConnection;

        public bool UseWebRTC { get { return _rtcPeerConnection != null; } }

        public bool UseSdp { get { return UseWebRTC && _rtcPeerConnection.UseSdp; } }

        public bool UseServerMute { get { return UseSdp; } } // these are the same for now

        public uint SSRC
        {
            get { return (uint)VoiceSessionMember.ClientID; }
        }

        public uint VideoSSRC
        {
            get { return 0x80000000 + (uint)VoiceSessionMember.ClientID; }
        }

        string IRTCStream.MSID
        {
            get { return "u" + (uint)VoiceSessionMember.ClientID; }
        }

        uint IRTCStream.SamplesPerPacket
        {
            get
            {
                var codec = VoiceSessionMember.CodecInfo;
                return codec.SampleRate*codec.PacketMilliseconds/1000;
            }
        }

        void IRTCStream.ProcessRtp(ByteBuffer buf)
        {
            // Track source SSRC for RTCP purposes
            var sourceSSRC = buf.ToUInt32BE(8);

            // Overwrite the SSRC so we can just copy this buffer to other RTC peers
            if (RTCRelayServer.IsAudioPacket(buf))
            {
                AudioSourceSSRC = sourceSSRC;
                buf.SetBytesBE((uint)SSRC, 8);
            }
            else
            {
                VideoSourceSSRC = sourceSSRC;
                buf.SetBytesBE((uint)VideoSSRC, 8);
            }

            Parent.BroadcastTransmit(this, buf);
        }

        void IRTCStream.ProcessRtcp(ByteBuffer buf)
        {
            Parent.BroadcastControl(this, buf);
        }

        private void TranslateRtcp(ByteBuffer buf)
        {
            // Loop through all RTCP messages in this packet
            var offset = 0;
            while (offset < buf.Count)
            {
                var fmt = buf[offset] & 0x1F;
                var pt = buf[offset + 1];
                var sourceSSRC = buf.ToUInt32BE(offset + 4);
                var len = (buf.ToUInt16BE(offset + 2) + 1)*4;

                switch (pt)
                {
                    case 200: // Sender Report
                    case 201: // Receiver Report
                    case 202: // SDES
                    {
                        // We don't care about these right now
                        break;
                    }

                    case 205: // Transport layer Feedback
                    {
                        var targetSSRC = buf.ToUInt32BE(offset + 8);
                        if (fmt == 1) // NACK
                        {
                            _rtcPeerConnection.ForwardNack(new ByteBuffer(buf.Buffer, buf.Offset + offset, len));
                        }
#if CONFIG_STAGING || CONFIG_DEBUG
                        else
                        {
                            Logger.Debug(string.Format("RTPFB: {0} {1}-{2}", fmt, sourceSSRC, targetSSRC));
                        }
#endif
                        break;
                    }

                    case 206: // Payload-specific Feedback
                    {
                        var targetSSRC = buf.ToUInt32BE(offset + 8);
                        if (fmt == 1) // PLI
                        {
                            _rtcPeerConnection.SendPli();
                        }
#if CONFIG_STAGING || CONFIG_DEBUG
                        else if (fmt == 4) // FIR
                        {
                            // TODO
                            Logger.Debug(string.Format("PSFB: FIR {1}-{2}", fmt, sourceSSRC, targetSSRC));
                        }
                        else if (fmt == 15) // Application-specific data
                        {
                            // Don't log these
                        }
                        else
                        {
                            Logger.Debug(string.Format("PSFB: {0} {1}-{2}", fmt, sourceSSRC, targetSSRC));
                        }
#endif
                        break;
                    }

                    default:
                    {
#if CONFIG_STAGING || CONFIG_DEBUG
                        Logger.Debug(string.Format("RTCP({0}): {1} {2}", pt, fmt, sourceSSRC));
#endif
                        break;
                    }
                }

                offset += len;
            }
        }

        public uint AudioSourceSSRC
        {
            get { return UseWebRTC ? _rtcPeerConnection.AudioSourceSSRC : 0; }
            set { if (UseWebRTC) { _rtcPeerConnection.AudioSourceSSRC = value; } }
        }

        public uint VideoSourceSSRC
        {
            get { return UseWebRTC ? _rtcPeerConnection.VideoSourceSSRC : 0; }
            set { if (UseWebRTC) { _rtcPeerConnection.VideoSourceSSRC = value; } }
        }

        private bool RequiresSSRCTranslation
        {
            get { return UseWebRTC && AudioSourceSSRC != SSRC; }
        }

        public void SendOffer()
        {
            if (UseSdp) // sanity check
            {
                var relayServer = VoiceServer.Instance.RelayServer;
                var sdp = _rtcPeerConnection.CreateOffer(relayServer.Sockets, Parent.Sessions);
                SendContract(new SessionDescriptionOffer { Description = sdp });
            }
        }

        public void SetAnswer(SessionDescriptionAnswer answer)
        {
            if (UseSdp) // sanity check
            {
                _rtcPeerConnection.SetAnswer(answer.Description);
            }
        }

        private RTCSocketPair _sockets;
        private readonly IPEndPoint _sender = new IPEndPoint(IPAddress.Any, 0);

        public void UpdateSockets(RTCSocketPair sockets, IPEndPoint sender)
        {
            _sockets = sockets;
            _sender.Address = sender.Address;
            _sender.Port = sender.Port;

            if (UseWebRTC)
            {
                _rtcPeerConnection.UpdateEndPoint(sender, true);
            }
        }

        public void SendRtpPacket(VoiceSession sender, ByteBuffer buf)
        {
            if (UseWebRTC)
            {
                _rtcPeerConnection.SendRtp(buf);
            }
            else if (RTCRelayServer.IsAudioPacket(buf))
            {
                UdpChannel.StartSend(_sender, GetVoicePacket(sender, buf));
            }
        }

        public void SendRtcpPacket(VoiceSession sender, ByteBuffer buf)
        {
            if (!UseWebRTC) // Only WebRTC clients care about these
            {
                return;
            }

            // If the client isn't receiving video, don't forward control packets
            if (!HasVideo || !CanSpeak || IsModDeafened)
            {
                return;
            }

            if (RequiresSSRCTranslation)
            {
                // Translate and forward NACK and PLI at least
                TranslateRtcp(buf);
            }
            else
            {
                _rtcPeerConnection.SendRtcp(buf);
            }
        }

        private ByteBuffer GetVoicePacket(VoiceSession sender, ByteBuffer buffer)
        {
            var headerLen = Srtp.GetHeaderLength(buffer);
            var voiceDataLen = buffer.Count - headerLen;

            //var seqNum2 = buffer.ToUInt16BE(2);
            var timestamp = buffer.ToUInt32BE(4);

            var socket = sender.ServerSocket;
            var clientID = socket.ClientID;
            var userID = sender.VoiceSessionMember.ClientID;
            var seqNum = timestamp / ((IRTCStream)sender).SamplesPerPacket;
            var endPoint = _sender;

            var resultLength = MessageHeader.HeaderSize + TransmitNotification.FixedSize + voiceDataLen;
            var result = new ByteBuffer(new byte[resultLength], 0, resultLength);

            // MessageHeader
            result.SetBytesLE(clientID, 0);
            result.SetBytesLE(TransmitNotification.FixedSize + voiceDataLen,  4);
            result.SetBytesLE(TransmitNotification.MessageType, 8);
            result.SetBytesLE(TransmitNotification.IsSerialized, 12);
            var offset = MessageHeader.HeaderSize;

            // TransmitNotification
            result.SetBytesLE(userID, offset);
            result.SetBytesLE(seqNum, offset + 4);
            result.SetBytesLE(0L, offset + 8);
            result.SetBytesLE(voiceDataLen, offset + 16);
            offset += TransmitNotification.FixedSize;

            // Voice data
            buffer.BlockCopy(headerLen, result, offset, voiceDataLen);

            return result;
        }

        #endregion
    }
}
