﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Threading;
using Curse.Logging;
using Curse.SocketInterface;
using Curse.SocketServer;
using Microsoft.ServiceModel.WebSockets;
using Curse.WebSocketInterface;

namespace Curse.Voice.HostRuntime
{
    public class VoiceWebSocketService : WebSocketService
    {
        private readonly BaseWebSocketServiceInterface _socketInterface;

        public VoiceWebSocketService()
        {
            try
            {
                if (VoiceServer.Instance.ConnectionCount >= HostRuntimeConfiguration.Instance.MaxConnections)
                {
                    Close();
                    return;
                }

                var clientID = VoiceServer.Instance.NextClientID();
                _socketInterface = new BaseWebSocketServiceInterface(this, IPAddress.None, clientID);

                if (!_clients.TryAdd(clientID, _socketInterface))
                {
                    Logger.Warn("Failed to add client to collection, idle cleanup won't see it");
                }

            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to create web socket");
                Close();
            }
        }

        public override void OnOpen()
        {
            try
            {
                _socketInterface.IsConnected = true;
            }
            catch (Exception ex)
            {
                Logger.Error(ex);
            }
        }

        public override void OnMessage(string message)
        {
            try
            {
                _socketInterface.ProcessIncomingMessage(message);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process incoming message!");
            }
        }
        
        protected override void OnClose()
        {
            try
            {
                BaseWebSocketServiceInterface value;
                _clients.TryRemove(_socketInterface.ClientID, out value);

                var session = _socketInterface.Session;
                if (session != null)
                {
                    session.Disconnect();
                }

                _socketInterface.IsConnected = false;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to cleanly close web socket");
            }
        }

        #region Idle Cleanup

        private static readonly ConcurrentDictionary<int, BaseWebSocketServiceInterface> _clients =
            new ConcurrentDictionary<int, BaseWebSocketServiceInterface>();

        public static ISocketInterface Lookup(int clientID)
        {
            BaseWebSocketServiceInterface result;
            _clients.TryGetValue(clientID, out result);
            return result;
        }

        /// <summary>
        /// This must be called after VoiceServer.Instance.Start()
        /// </summary>
        public static void StartCleanupThread()
        {
            new Thread(IdleSocketCleanup) { IsBackground = true, Name = "IdleWebSocketCleanup" }.Start();
        }

        private static void IdleSocketCleanup()
        {
            while (VoiceServer.Instance.State == SocketServerState.Started)
            {
                try
                {
                    Thread.Sleep(1000);

                    if (VoiceServer.Instance.State != SocketServerState.Started)
                    {
                        return;
                    }

                    DisconnectIdleSockets();
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Unhandled exception in IdleSocketCleanup");
                }

            }

            Logger.Info("Idle socket cleanup thread has completed.", new { VoiceServer.Instance.State });
        }

        #endregion Idle Cleanup

        private static void DisconnectIdleSockets()
        {
            // Get all socket interfaces that need a health check
            var socketsToCheck = _clients.Values.Where(client => client.NeedsHealthCheck()).ToArray();

            if (socketsToCheck.Length == 0)
            {
                return;
            }

#if SOCKET_LOGGING
            Logger.Trace("Health checking " + socketsToCheck.Length.ToString("###,##0") + " sockets...");
#endif

            var socketsToDisconnect = new List<ISocketInterface>();
            var socketsNeverConnected = 0;
            var socketsNeverAuthenticated = 0;
            var socketsWithNoSession = 0;
            var socketsUnhealthy = 0;

            foreach (var client in socketsToCheck)
            {
                var needsDisconnect = true;
                if (!client.IsConnected)
                {
                    ++socketsNeverConnected;
                }
                else if (client.Session == null)
                {
                    ++socketsNeverAuthenticated;
                }
                else if (!client.IsAuthenticated)
                {
                    ++socketsWithNoSession;
                }
                else if (!client.IsSocketHealthy())
                {
                    ++socketsUnhealthy;
                }
                else
                {
                    needsDisconnect = false;
                }

                if (needsDisconnect)
                {
                    socketsToDisconnect.Add(client);
                }
            }


            if (socketsToDisconnect.Count == 0)
            {
                return;
            }

            SocketServer.SocketServer.SocketStats.IncrementIdleSocketCount(socketsToDisconnect.Count, socketsNeverConnected, socketsNeverAuthenticated, socketsWithNoSession, socketsUnhealthy);

            Logger.Trace("Disconnecting " + socketsToDisconnect.Count.ToString("###,##0") + " idle clients...");

            foreach (var client in socketsToDisconnect)
            {
                try
                {
                    client.RaiseDisconnect(SocketDisconnectReason.IdleTimeout);
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to disconnect client!");
                }

                if (_clients.ContainsKey(client.ClientID))
                {
                    BaseWebSocketServiceInterface removed;
                    if (!_clients.TryRemove(client.ClientID, out removed))
                    {
                        Logger.Error("Failed to remove client from client dictionary!");
                    }
                }
            }
        }
    }
}
