﻿using System;
using System.Collections.Concurrent;
using System.IO;
using System.Threading;
using Curse.Logging;
using Newtonsoft.Json;
using System.ServiceModel;
using System.Collections.Generic;
using System.Text;
using Curse.Voice.Logging.CurseVoiceService;

namespace Curse.Voice.Logging
{
    public class VoiceServiceLogWriter : ILogWriter
    {        
        private static bool _isAlive;
        private JsonSerializer _serializer;
        private readonly ConcurrentQueue<LogFileEntry> _entries = new ConcurrentQueue<LogFileEntry>();
        private readonly string _serviceUrl;
        private readonly string _serviceApiKey;
        private readonly LogDataPayloadType _appType;
        private readonly string _appExternalID;
        private DateTime _startTime = DateTime.UtcNow;
        private DateTime _lastCalledHome = DateTime.UtcNow.AddHours(-2);
        private readonly TimeSpan _callHomeFrequency = TimeSpan.FromHours(1);

        public VoiceServiceLogWriter(string serviceUrl, LogDataPayloadType appType, string serviceApiKey, string appExternalID = null)
        {
            _serviceUrl = serviceUrl;
            _serviceApiKey = serviceApiKey;
            _appType = appType;
            _appExternalID = appExternalID;
            _isAlive = true;

            new Thread(UploadThread) { Name = "UploadThread", IsBackground = true }.Start();            
        }

        public void Shutdown()
        {
            _isAlive = false;
            UploadToService();
        }

        public void Write(LogFileEntry entry)
        {            
            if (_entries.Count > 5000)
            {
                return;
            }
            _entries.Enqueue(entry);            
        }

        public void Initialize(JsonSerializer serializer)
        {
            _serializer = serializer;
        }

        public void UploadToService()
        {            
            var newEntries = new List<LogFileEntry>();
               
            var count = _entries.Count;
            if (count >= 5000)
            {
                for(var i=0;i<5000;i++)
                {
                    LogFileEntry entry;
                    if (!_entries.TryDequeue(out entry))
                    {
                        break;
                    }
                }

                _entries.Enqueue(new LogFileEntry
                {
                    Level = LogLevel.Warn,
                    Message = "Log data was cleared, due to exceeding limits.",
                    Timestamp = DateTime.UtcNow,
                    Data = new { NumEntries = count }
                });
            }

            // Upload 1,000 entries at a time
            while (_entries.Count > 0 && newEntries.Count < 1000)
            {
                LogFileEntry entry;
                if (!_entries.TryDequeue(out entry))
                {
                    break;
                }
                    
                if (entry == null)
                {
                    break;
                }

                newEntries.Add(entry);
            }
                

            if (newEntries.Count == 0)
            {
                return;
            }

            var sb = new StringBuilder();

            using (var writer = new StringWriter(sb))
            {
                _serializer.Serialize(writer, newEntries.ToArray());
            }

            var success = false;
            try
            {
                TrySubmitLogdata(sb.ToString());
                success = true;
            }
            catch (TimeoutException ex)
            {
                Logger.Warn(ex, "Timeout trying to submit log data to service!");
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to submit log data to service!");
            }

            if (success) return;

            // Something failed. Add the entries back for uploading later.
                
            foreach (var entry in newEntries)
            {
                _entries.Enqueue(entry);
            }                
            
        }

        private void TrySubmitLogdata(string newLogData)
        {
            using (var client = new CurseVoiceServiceClient(GetBasicHttpBinding(30), new EndpointAddress(_serviceUrl)))
            {
                var logFilePayload = new LogDataPayload
                {
                    ApiKey = _serviceApiKey, 
                    ExternalID = _appExternalID,
                    HostName = Environment.MachineName,
                    Type = _appType,
                    SerializedLogEntries = newLogData
                };

                client.UploadLogData(logFilePayload);
            }
        }

        public static BasicHttpBinding GetBasicHttpBinding(int sendTimeoutSeconds = 10)
        {
            var binding = new BasicHttpBinding()
            {
                OpenTimeout = TimeSpan.FromSeconds(5),
                SendTimeout = TimeSpan.FromSeconds(sendTimeoutSeconds),
                ReceiveTimeout = TimeSpan.FromSeconds(10),

            };

            return binding;
        }

        private void CallHome()
        {
            if (DateTime.UtcNow.Subtract(_lastCalledHome) < _callHomeFrequency)
            {
                return;
            }

            _lastCalledHome = DateTime.UtcNow;
            
            Logger.Info("Voice server calling home. My name is " + Environment.MachineName + " and I have been online since " + _startTime);
        }

        private void UploadThread()
        {
            while (true)
            {
                Thread.Sleep(5000);

                if (!_isAlive)
                {
                    return;
                }

                try
                {
                    CallHome();
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to call home!");
                }

                try
                {
                    UploadToService();
                }
                catch (ThreadAbortException) { }
                catch (Exception ex)
                {
                    if (ex is AggregateException)
                    {
                        ex = (ex as AggregateException).Flatten();
                    }

                    Logger.Error(ex, "Upload Thread Error");
                }

            }
        }
    }
}
