﻿using System;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.Net;
using System.Net.Http;
using System.Net.Http.Headers;
using System.Text;
using Newtonsoft.Json;

namespace Curse.ServiceClients
{
    public abstract class BaseWebServiceClient
    {
        private readonly string _url;

        public string Url
        {
            get { return _url; }
        }

        protected BaseWebServiceClient(string url)
        {
            if (string.IsNullOrEmpty(url))
            {
                throw new ArgumentNullException("url", "Url must be supplied for service: " + GetType().Name);
            }

            _url = url;
        }

        public static ServicesConfiguration Configuration = new ServicesConfiguration();

        IDictionary<string, string> GetHeaders()
        {
            if (ServiceAuthentication.Token != null)
            {
                return new Dictionary<string, string> { { "AuthenticationToken", ServiceAuthentication.Token } };
            }
            return null;
        }

        protected virtual HttpClient GetHttpClient()
        {
            HttpClientHandler handler;
            if (Configuration.CustomHttpHandler != null)
            {
                handler = (HttpClientHandler)Activator.CreateInstance(Configuration.CustomHttpHandler);
            }
            else
            {
                handler = new HttpClientHandler();
            }
            handler.UseProxy = !Configuration.DisableProxyDetection;

            var httpClient = new HttpClient(handler) { Timeout = TimeSpan.FromSeconds(Configuration.TimeoutSeconds) };

            if (Configuration.UserAgent != null)
            {
                httpClient.DefaultRequestHeaders.TryAddWithoutValidation("User-Agent", Configuration.UserAgent);
            }
            if (Configuration.ContentType != null)
            {
                httpClient.DefaultRequestHeaders.Accept.Add(
                    new MediaTypeWithQualityHeaderValue(Configuration.ContentType));
            }

            var header = GetHeaders();
            if (header != null)
            {
                foreach (var kvp in header)
                {
                    httpClient.DefaultRequestHeaders.TryAddWithoutValidation(kvp.Key, kvp.Value);
                }
            }

            return httpClient;
        }

        private string GetUrl(string action)
        {
            return _url + "/" + action;
        }

        protected virtual ServiceResponse Patch(string action, object body)
        {
            throw new NotImplementedException();
        }


        protected virtual ServiceResponse<TResponse> Post<TResponse>(string action)
        {
            var url = GetUrl(action);

            try
            {
                HttpResponseMessage response = null;
                
                using (new SimpleTimer(action, "Request to " + url))
                {
                    using (var httpClient = GetHttpClient())
                    {
                        response = httpClient.PostAsync(url, null).Result;
                    }
                }

                if ((int)response.StatusCode < 200 || (int)response.StatusCode >= 300)
                {
                    return new ServiceResponse<TResponse>(response.StatusCode);
                }

                TResponse value;
                using (new SimpleTimer(action, "Deserialization"))
                {
                    var s = response.Content.ReadAsStringAsync().Result;
                    value = JsonConvert.DeserializeObject<TResponse>(s);
                }

                return new ServiceResponse<TResponse>(response.StatusCode, value);
            }
            catch  (Exception ex)
            {
                ServicesConfiguration.LogWarning(ex, "Service call failed to: " + url);
                return new ServiceResponse<TResponse>(HttpStatusCode.InternalServerError);
            }
            
        }


        protected virtual ServiceResponse<TResponse> Post<TResponse>(string action, object body)
        {
            var url = GetUrl(action);

            try
            {
                string json;

                using (new SimpleTimer(action, "Serialization"))
                {
                    json = JsonConvert.SerializeObject(body);
                }

                HttpResponseMessage response = null;
               
                using (new SimpleTimer(action, "Request to " + url))
                {
                    using (var httpClient = GetHttpClient())
                    {
                        response =
                            httpClient.PostAsync(url, new StringContent(json, Encoding.UTF8, Configuration.ContentType))
                                .Result;
                    }
                }

                if ((int)response.StatusCode < 200 || (int)response.StatusCode >= 300)
                {
                    return new ServiceResponse<TResponse>(response.StatusCode);
                }

                TResponse value;
                using (new SimpleTimer(action, "Deserialization"))
                {
                    var s = response.Content.ReadAsStringAsync().Result;
                    value = JsonConvert.DeserializeObject<TResponse>(s);
                }

                return new ServiceResponse<TResponse>(response.StatusCode, value);
            }
            catch (Exception ex)
            {
                ServicesConfiguration.LogWarning(ex, "Service call failed to: " + url);
                return new ServiceResponse<TResponse>(HttpStatusCode.InternalServerError);
            }
            
        }

        protected virtual ServiceResponse Post(string action)
        {
            var url = GetUrl(action);

            try
            {
                HttpResponseMessage response = null;
                
                using (new SimpleTimer(action, "Request to " + url))
                {
                    using (var httpClient = GetHttpClient())
                    {
                        response =
                            httpClient.PostAsync(url, null)
                                .Result;
                    }
                }

                return new ServiceResponse(response.StatusCode);
            }
            catch (Exception ex)
            {
                ServicesConfiguration.LogWarning(ex, "Service call failed to: " + url);
                return new ServiceResponse(HttpStatusCode.InternalServerError);
            }

        }

        protected virtual ServiceResponse Post(string action, object body)
        {
            var url = GetUrl(action);

            try
            {
                string json;

                using (new SimpleTimer(action, "Serialization"))
                {
                    json = JsonConvert.SerializeObject(body);
                }

                HttpResponseMessage response = null;
                
                using (new SimpleTimer(action, "Request to " + url))
                {
                    using (var httpClient = GetHttpClient())
                    {
                        response =
                            httpClient.PostAsync(url, new StringContent(json, Encoding.UTF8, Configuration.ContentType))
                                .Result;
                    }
                }

                return new ServiceResponse(response.StatusCode);
            }
            catch (Exception ex)
            {
                ServicesConfiguration.LogWarning(ex, "Service call failed to: " + url);
                return new ServiceResponse(HttpStatusCode.InternalServerError);
            }
        }

        protected virtual ServiceResponse Delete(string action)
        {
            var url = GetUrl(action);

            try
            {
                HttpResponseMessage response = null;
               
                using (new SimpleTimer(action, "Request to " + url))
                {
                    using (var httpClient = GetHttpClient())
                    {
                        response = httpClient.DeleteAsync(url).Result;
                    }
                }

                return new ServiceResponse(response.StatusCode);
            }
            catch (Exception ex)
            {
                ServicesConfiguration.LogWarning(ex, "Service call failed to: " + url);
                return new ServiceResponse(HttpStatusCode.InternalServerError);
            }           
        }

        protected virtual ServiceResponse<TResponse> Delete<TResponse>(string action)
        {
            var url = GetUrl(action);

            try
            {
                HttpResponseMessage response = null;

                using (new SimpleTimer(action, "Request to " + url))
                {
                    using (var httpClient = GetHttpClient())
                    {
                        response = httpClient.DeleteAsync(url).Result;
                    }
                }

                if ((int)response.StatusCode < 200 || (int)response.StatusCode >= 300)
                {
                    return new ServiceResponse<TResponse>(response.StatusCode);
                }

                TResponse value;
                using (new SimpleTimer(action, "Deserialization"))
                {
                    var s = response.Content.ReadAsStringAsync().Result;
                    value = JsonConvert.DeserializeObject<TResponse>(s);
                }

                return new ServiceResponse<TResponse>(response.StatusCode, value);
            }
            catch (Exception ex)
            {
                ServicesConfiguration.LogWarning(ex, "Service call failed to: " + url);
                return new ServiceResponse<TResponse>(HttpStatusCode.InternalServerError);
            }
        }

        protected virtual ServiceResponse<TResponse> Get<TResponse>(string action)
        {
            var url = GetUrl(action);

            try
            {                
                using (var httpClient = GetHttpClient())
                {                    
                    HttpResponseMessage response;

                    using (new SimpleTimer(action, "Request to " + url))
                    {
                        response = httpClient.GetAsync(url).Result;
                    }

                    if ((int)response.StatusCode < 200 || (int)response.StatusCode >= 300)
                    {
                        return new ServiceResponse<TResponse>(response.StatusCode);
                    }

                    TResponse value;
                    using (new SimpleTimer(action, "Deserialization"))
                    {
                        var s = response.Content.ReadAsStringAsync().Result;
                        value = JsonConvert.DeserializeObject<TResponse>(s);
                    }

                    return new ServiceResponse<TResponse>(response.StatusCode, value);
                }
            }
            catch (Exception ex)
            {
                ServicesConfiguration.LogWarning(ex, "Service call failed to: " + url);
                return new ServiceResponse<TResponse>(HttpStatusCode.InternalServerError);
            }

        }

        protected virtual NameValueCollection GetCustomHeaders()
        {
            return null;
        }
    }
}