﻿using System;
using Curse.ServiceClients.Contracts;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using NSubstitute;

namespace Curse.Voice.UpdateDeployment.Tests
{
    [TestClass]
    public class ClientUpdateDeploymentTaskTests
    {
        private MockConsole Console { get; set; }
        private IDeployHelper DeployHelper { get; set; }
        private ClientUpdateDeploymentTask Deploy { get; set; }

        [TestInitialize]
        public void Initialize()
        {
            Console = new MockConsole();
            DeployHelper = Substitute.For<IDeployHelper>();
            Deploy = new ClientUpdateDeploymentTask()
            {
                DeployHelper = DeployHelper,
                Console = Console,
            };
            ConsoleHelper.Console = Console;
        }

        [TestMethod]
        public void RegisterClientVersion_ShouldRegisterVersion_WhenUserConfirms()
        {
            Deploy.ReleaseType = "Beta";
            Deploy.ApplicationVersion = "7.6.6435.42098";
            Deploy.CentralServiceApiKey = "asdf";
            Deploy.AppDisplayName = "TwitchDesktop";
            Console.TestSetUserInput(Keys.Y); // Yes, register

            Deploy.RegisterClientVersion();

            DeployHelper.Received().RegisterClientVersion("asdf", "TwitchDesktop", ReleaseType.Beta, DevicePlatform.Windows, "7.6.6435.42098", Arg.Any<DateTime>());
        }

        [TestMethod]
        public void RegisterClientVersion_ShouldNotRegisterVersion_WhenUserCancels()
        {
            Deploy.ReleaseType = "Beta";
            Console.TestSetUserInput(Keys.N); // No, do not register

            Deploy.RegisterClientVersion();

            DeployHelper.DidNotReceive().RegisterClientVersion(Arg.Any<string>(), Arg.Any<string>(), Arg.Any<ReleaseType>(), Arg.Any<DevicePlatform>(), Arg.Any<string>(), Arg.Any<DateTime>());
        }

        [TestMethod]
        public void RegisterClientVersion_ShouldAskUserToRetry_WhenRegistrationFails()
        {
            Deploy.ReleaseType = "Beta";
            DeployHelper.WhenForAnyArgs(h => h.RegisterClientVersion(Arg.Any<string>(), Arg.Any<string>(), Arg.Any<ReleaseType>(), Arg.Any<DevicePlatform>(), Arg.Any<string>(), Arg.Any<DateTime>())).Do(h => { throw new Exception(); });
            Console.TestSetUserInput(Keys.Y); // Yes, register
            Console.TestSetUserInput(Keys.Y); // Yes, try again
            Console.TestSetUserInput(Keys.Y); // Yes, register
            Console.TestSetUserInput(Keys.N); // No, do not try again
            Console.TestSetUserInput(Keys.Y); // Yes, continue deploy

            Deploy.RegisterClientVersion();

            DeployHelper.Received(2).RegisterClientVersion(Arg.Any<string>(), Arg.Any<string>(), Arg.Any<ReleaseType>(), Arg.Any<DevicePlatform>(), Arg.Any<string>(), Arg.Any<DateTime>());
        }

        [TestMethod]
        [ExpectedException(typeof(Exception))]
        public void RegisterClientVersion_ShouldCancelDeploy_WhenRegistrationFails()
        {
            Deploy.ReleaseType = "Beta";
            Deploy.ApplicationVersion = "7.6.6435.42098";
            Deploy.CentralServiceApiKey = "asdf";
            Deploy.AppDisplayName = "TwitchDesktop";
            DeployHelper.ThrowsForAnyArgs(h => h.RegisterClientVersion(Arg.Any<string>(), Arg.Any<string>(), Arg.Any<ReleaseType>(), Arg.Any<DevicePlatform>(), Arg.Any<string>(), Arg.Any<DateTime>()));
            Console.TestSetUserInput(Keys.Y); // Yes, register
            Console.TestSetUserInput(Keys.N); // No, do not try again
            Console.TestSetUserInput(Keys.N); // No, end deploy

            Deploy.RegisterClientVersion();
        }


        [TestMethod]
        public void TagClientVersion_ShouldTag_WhenUserConfirms()
        {
            Deploy.ReleaseType = "Beta";
            Deploy.ApplicationVersion = "7.6.6435.42098";
            Deploy.SolutionPath = @"C:\Projects\desktop\Curse.Radium.Windows.sln";
            Console.TestSetUserInput(Keys.Y); // Yes, tag

            Deploy.TagClientVersion();

            DeployHelper.Received(1).TagRepo(@"C:\Projects\desktop", "7.6.6435.42098");
            DeployHelper.Received(1).TagRepo(@"C:\Projects\laguna", "7.6.6435.42098");
        }

        [TestMethod]
        public void TagClientVersion_ShouldNotTag_WhenUserCancels()
        {
            Deploy.ReleaseType = "Beta";
            Deploy.ApplicationVersion = "7.6.6435.42098";
            Deploy.SolutionPath = @"C:\Projects\desktop\Curse.Radium.Windows.sln";
            Console.TestSetUserInput(Keys.N); // No, do not tag

            Deploy.TagClientVersion();

            DeployHelper.DidNotReceive().TagRepo(Arg.Any<string>(), Arg.Any<string>());
        }

        [TestMethod]
        public void TagClientVersion_ShouldContinueDeploy_WhenRegistrationFails()
        {
            Deploy.ReleaseType = "Beta";
            Deploy.ApplicationVersion = "7.6.6435.42098";
            Deploy.SolutionPath = @"C:\Projects\desktop\Curse.Radium.Windows.sln";
            DeployHelper.ThrowsForAnyArgs(h => h.TagRepo(Arg.Any<string>(), Arg.Any<string>()));
            Console.TestSetUserInput(Keys.Y); // Yes, tag

            Deploy.TagClientVersion();

            DeployHelper.Received(1).TagRepo(@"C:\Projects\desktop", "7.6.6435.42098");
            DeployHelper.DidNotReceive().TagRepo(@"C:\Projects\laguna", "7.6.6435.42098");
        }
        
        private static class Keys
        {
            public static readonly ConsoleKeyInfo N = new ConsoleKeyInfo('n', ConsoleKey.N, false, false, false);
            public static readonly ConsoleKeyInfo Y = new ConsoleKeyInfo('y', ConsoleKey.Y, false, false, false);
        }
    }

    public static class Extensions
    {
        public static void ThrowsForAnyArgs<T>(this T substitute, Action<T> substituteCall) where T : class
        {
            substitute.WhenForAnyArgs(substituteCall).Do(h => { throw new Exception(); });
        }
    }
}
