﻿using System.Diagnostics;
using Curse.MSBuild.Deployment;
using Curse.Voice.Helpers;
﻿using Curse.MSBuild.Deployment;
using Curse.Voice.UpdateDeployment.HostUpdateService;
using ICSharpCode.SharpZipLib.Zip;
using Microsoft.Build.Framework;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.ServiceModel;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Data.SqlClient;
using System.Net;
using Amazon.Runtime;
using Amazon.S3;
using Amazon.S3.Transfer;
using Curse.CloudFlare;
using Newtonsoft.Json;
using Curse.ServiceClients;

namespace Curse.Voice.UpdateDeployment
{
    public class ClientUpdateDeploymentTask : ITask
    {
        public IDeployHelper DeployHelper { get; set; } = new DeployHelper(new MongooseClient("https://mongoose.cpe.wtf"));
        public IConsole Console { get; set; } = new ConsoleProxy();

        public IBuildEngine BuildEngine { get; set; }
        public ITaskHost HostObject { get; set; }


        public string AppZipPath;

        [Required]
        public string AppDisplayName { get; set; }

        [Required]
        public string SolutionPath { get; set; }

        [Required]
        public string RootSourcePath { get; set; }

        [Required]
        public string ApplicationVersion { get; set; }

        [Required]
        public string ApplicationSourcePath { get; set; }

        [Required]
        public string InstallerSourcePath { get; set; }

        [Required]
        public string NetworkInstallerSourcePath { get; set; }

        [Required]
        public string NetworkInstallerUrl { get; set; }

        [Required]
        public string ReleaseType { get; set; }

        [Required]
        public string[] UpdateServers { get; set; }

        [Required]
        public bool DeployInstaller { get; set; }

        [Required]
        public bool PurgeCache { get; set; }

        public string InstallerUrl { get; set; }
        public string UpdateUrl { get; set; }
        public string ManifestUrl { get; set; }

        public string BinPath { get; set; }

        [Required]
        public string CentralServiceUrl { get; set; }

        [Required]
        public string CentralServiceApiKey { get; set; }

        public bool AntivirusWhitelist { get; set; }

        private VoiceHostEnvironment _hostEnvironment;

        private List<string> _whitelistFiles = new List<string>(); 
      
        public bool Execute()
        {

            DeployStep.Initialize();

            var secretPath = @"C:\desktop_secrets.json";
            if (!File.Exists(secretPath))
            {
                throw new Exception("Missing deploy secrets");
            }

            var secretJson = File.ReadAllText(secretPath);
            var secret = JsonConvert.DeserializeObject<DesktopDeploySecret>(secretJson);

            BaseServices.Urls = new ServiceReleaseUrls();

            DeployStep.RegisterStep("Validating Configuration", () => true, () =>
            {
                FileInfo fi = new FileInfo(@"C:\Program Files (x86)\PsExec\psexec.exe");

                if (!fi.Exists)
                {
                    Console.WriteLine(@"You are missing a local copy of PsExec. Please put it at C:\Program Files (x86)\PsExec\psexec.exe");
                    throw new Exception("Missing file.");
                }


                // Config validation
                _hostEnvironment = (VoiceHostEnvironment)Enum.Parse(typeof(VoiceHostEnvironment), ReleaseType);

                Version clientVersion = null;
                if (!Version.TryParse(ApplicationVersion, out clientVersion))
                {                    
                    throw new Exception("Unable to parse client version '" + ApplicationVersion + "'");
                }

                if (DeployInstaller)
                {
                    // Installer Validation
                    fi = new FileInfo(InstallerSourcePath);

                    if (!fi.Exists)
                    {
                        Console.WriteLine(@"We could not locate the installer at: " + InstallerSourcePath);
                        throw new Exception("Missing file.");
                    }

                    if (DateTime.UtcNow.Subtract(fi.LastWriteTimeUtc) > TimeSpan.FromHours(4))
                    {
                        throw new Exception("Out of date file '" + fi.FullName + "'");
                    }
                }
                else
                {
                    Console.WriteLine("Skipping installer validation, as it is not being deployed.");
                }

                // Update Binaries
                var di = new DirectoryInfo(ApplicationSourcePath);

                var updateBinaryFiles = di.GetFiles();
                if (!updateBinaryFiles.Any(p => p.Name.StartsWith("CurseClientUpdater.exe")))
                {
                    Console.WriteLine(@"The update binary files are missing CurseClientUpdater.exe");
                    throw new Exception("Unable to locate 'CurseClientUpdater.exe' in '" + di.FullName + "'");
                }

                if (!updateBinaryFiles.Any(p => p.Name.StartsWith("Curse.exe")))
                {
                    throw new Exception("Unable to locate 'Curse.exe' in '" + di.FullName + "'");                    
                }

                if (!updateBinaryFiles.Any(p => p.Name.StartsWith("Twitch.exe")))
                {
                    throw new Exception("Unable to locate 'Twitch.exe' in '" + di.FullName + "'");
                }

                // URLs
                if (PurgeCache)
                {
                    if (!Uri.IsWellFormedUriString(ManifestUrl, UriKind.Absolute))
                    {
                        throw new Exception("Invalid Manifest URL: " + ManifestUrl);
                    }

                    if (!Uri.IsWellFormedUriString(InstallerUrl, UriKind.Absolute))
                    {
                        throw new Exception("Invalid Installer URL: " + InstallerUrl);
                    }

                    if (!Uri.IsWellFormedUriString(UpdateUrl, UriKind.Absolute))
                    {
                        throw new Exception("Invalid Update URL: " + ManifestUrl);
                    }                    
                }

            });

            DeployStep.RegisterStep("Comparing changed files", () => AntivirusWhitelist, () =>
            {
                AppUpdate latestProducitonUpdate = null;
                AppUpdate currentUpdate = null;
                try
                {
                    byte[] manifestData = null;
                    using (var client = new WebClient())
                    {
                        manifestData = client.DownloadData(ManifestUrl);
                    }

                    var manifest = AppManifest.Deserialize(manifestData);
                    var version = manifest.Versions.FirstOrDefault(p => p.ReleaseType.ToString() == ReleaseType);

                    using (var client = new WebClient())
                    {
                        var data = client.DownloadData("http://updates.curseapp.net/windows/" + ReleaseType + "/" + version.VersionString +
                                            "/Update.xml");
                        latestProducitonUpdate = AppUpdate.Deserialize(data);
                    }
                }
                catch (Exception ex)
                {
                    throw new Exception("Failed to download manifest file.", ex);
                }

                if (latestProducitonUpdate == null)
                {
                    throw new Exception("No remote update file");
                }

                var localFile = File.ReadAllBytes(Path.Combine(ApplicationSourcePath, "Update.xml"));
                currentUpdate = AppUpdate.Deserialize(localFile);

                if (currentUpdate == null)
                {
                    throw new Exception("No local update file");
                }

                long savedBytes = 0;

                foreach (var file in currentUpdate.Files.Where(f => f.Name.EndsWith(".exe") || f.Name.EndsWith(".dll")))
                {
                    var remoteFile = latestProducitonUpdate.Files.FirstOrDefault(p => p.Name == file.Name);
                    if (remoteFile == null)
                    {
                        _whitelistFiles.Add(file.Name);
                        continue;
                    }

                    if (file.Hash != remoteFile.Hash)
                    {
                        _whitelistFiles.Add(file.Name);
                        continue;
                    }
                    var test = new FileInfo(Path.Combine(BinPath, file.Name));
                    if (test.Exists)
                    {
                        savedBytes += test.Length;
                    }
                }

                Console.WriteLine("Added " + _whitelistFiles.Count + " files to whitelist ... Saved " + ((savedBytes / 1024f) / 1024f) + " MB");
            });

            DeployStep.RegisterStep("Antivirus Whitelisting", () => AntivirusWhitelist, () =>
            {
                if (BinPath != null)
                {
                    var binPath = Path.GetFullPath(BinPath);

                    var endpoints = new List<FtpEndpoint>();
                    foreach (var whitelist in secret.Whitelist)
                    {
                        endpoints.Add(new FtpEndpoint(binPath, whitelist.Host, whitelist.Path, whitelist.Username, whitelist.Pasword));
                    }

                    foreach (var endpoint in endpoints)
                    {
                        try
                        {
                            FtpWhitelisting.UploadFiles(endpoint.SourceFolder, endpoint.Host, endpoint.Path, endpoint.Username, endpoint.Password, _whitelistFiles);
                        }
                        catch (Exception ex)
                        {
                            Console.ForegroundColor = ConsoleColor.Red;
                            Console.WriteLine("Failed: " + ex.Message);
                            Console.ResetColor();
                            if (ConsoleHelper.PromptUser("Antivirus upload failed. Continue anyway?", new[] { ConsoleKey.Y, ConsoleKey.N }) == ConsoleKey.N)
                            {
                                throw new Exception("Deploy cancelled!");
                            }
                        }
                    }
                }
            });

            DeployStep.RegisterStep("Deploy files to AWS", () => true, () =>
            {
                var client = new AmazonS3Client(new BasicAWSCredentials(secret.S3AccessKey, secret.S3SecretKey), Amazon.RegionEndpoint.USEast1);
                var util = new TransferUtility(client);

                var rootPath = RootSourcePath;

                if (!rootPath.EndsWith("\\"))
                {
                    rootPath = rootPath + "\\";
                }

                Console.Write("Copying Deploy to AWS ...");

                var dir = new DirectoryInfo(ApplicationSourcePath);
                foreach (var file in dir.GetFiles("*.*", SearchOption.AllDirectories))
                {
                    string relativePath = file.FullName.Replace(rootPath, "");
                    relativePath = "windows/" + relativePath.Replace("\\", "/");
                    util.Upload(file.FullName, "updates.curseapp.net", relativePath);
                }
                Console.WriteLine("Done!");
            });

            DeployStep.RegisterStep("Deploying Installer to AWS", () => DeployInstaller, () =>
            {
                if (ConsoleHelper.PromptUser("Are you ready to deploy the installer to AWS?", new[] { ConsoleKey.Y, ConsoleKey.N }) == ConsoleKey.N)
                {
                    throw new Exception("Deploy cancelled!");
                }
                var client = new AmazonS3Client(new BasicAWSCredentials(secret.S3AccessKey, secret.S3SecretKey), Amazon.RegionEndpoint.USEast1);
                var util = new TransferUtility(client);

                DeployInstallerS3(InstallerSourcePath, InstallerUrl, util);
                DeployInstallerS3(NetworkInstallerSourcePath, NetworkInstallerUrl, util);
            });

            DeployStep.RegisterStep("Deploying Update and Manifest File", () => true, () =>
            {
                if (ConsoleHelper.PromptUser("Are you ready to commit this deploy to AWS?", new[] { ConsoleKey.Y, ConsoleKey.N }) == ConsoleKey.N)
                {
                    throw new Exception("Deploy cancelled!");
                }

                var client = new AmazonS3Client(new BasicAWSCredentials(secret.S3AccessKey, secret.S3SecretKey), Amazon.RegionEndpoint.USEast1);
                var util = new TransferUtility(client);

                var di = new DirectoryInfo(ApplicationSourcePath);

                {
                    string sourcePath = Path.Combine(di.Parent.Parent.FullName, "Manifest.xml");
                    var relativePath = GetRelativePart(ManifestUrl);
                    if (relativePath.StartsWith("/"))
                    {
                        relativePath = relativePath.Remove(0, 1);
                    }
                    util.Upload(sourcePath, "updates.curseapp.net", relativePath);
                }

                {
                    string sourcePath = Path.Combine(di.FullName, "Update.xml");
                    var relativePath = GetRelativePart(UpdateUrl);
                    if (relativePath.StartsWith("/"))
                    {
                        relativePath = relativePath.Remove(0, 1);
                    }
                    util.Upload(sourcePath, "updates.curseapp.net", relativePath);
                }
            });

            DeployStep.RegisterStep("Registering Client Version", () => true, () =>
            {
                RegisterClientVersion();
            });

            DeployStep.RegisterStep("Purging CloudFront Cache", () => PurgeCache, () =>
            {
                var urls = new[] { GetRelativePart(UpdateUrl), GetRelativePart(ManifestUrl), GetRelativePart(InstallerUrl) };
                Console.Write("Purging CloudFront URLs at '" + GetTopLevelDomain(UpdateUrl) + "'... ");

                //E1Z399MZ0IX91C
                var success = CloudFrontHelper.InvalidateUrls("E1Z399MZ0IX91C", urls); // updates.curseapp.net
                Console.WriteLine(success ? "Success" : "Failed!");

                success = CloudFrontHelper.InvalidateUrls("E3PSEHGC8IN0L0", urls); // updates-backup.curseapp.net
                Console.WriteLine(success ? "Success" : "Failed!");
            });

            DeployStep.RegisterStep("Purging Cloudflare Cache", () => PurgeCache, () =>
            {
                ServicePointManager.SecurityProtocol = SecurityProtocolType.Tls11 | SecurityProtocolType.Tls12;
           
                CloudFlareApi.Initialize(secret.CloudFlareKey, secret.CloudFlareEmail);

                var updateDomain = GetTopLevelDomain(UpdateUrl);
                Console.Write("Purging Update URL at '" + UpdateUrl + "'... ");
                var success = CloudFlareApi.InvalidateUrl(updateDomain, UpdateUrl);
                Console.WriteLine(success ? "Success" : "Failed!");

                var manifestDomain = GetTopLevelDomain(ManifestUrl);
                Console.Write("Purging Manifest URL at '" + ManifestUrl + "'... ");
                success = CloudFlareApi.InvalidateUrl(manifestDomain, ManifestUrl);
                Console.WriteLine(success ? "Success" : "Failed!");

                var installerDomain = GetTopLevelDomain(InstallerUrl);
                Console.Write("Purging Installer URL at '" + InstallerUrl + "'... ");
                success = CloudFlareApi.InvalidateUrl(installerDomain, InstallerUrl);
                Console.WriteLine(success ? "Success" : "Failed!");
            });
            
            DeployStep.RegisterStep("Tag Version in GIT", () => true, () =>
            {
                TagClientVersion();
            });

            DeployStep.RunAll();
            return true;
        }

        private void DeployInstallerS3(string sourcePath, string installerUrl, TransferUtility util)
        {
            Console.Write("Copying Installer from: " + sourcePath + " ...");
            var file = new FileInfo(sourcePath);
            var relativePath = GetRelativePart(installerUrl);
            if (relativePath.StartsWith("/"))
            {
                relativePath = relativePath.Remove(0, 1);
            }
            util.Upload(file.FullName, "updates.curseapp.net", relativePath);
            Console.WriteLine("Installer deployed to: " + relativePath);
        }

        public void RegisterClientVersion()
        {
            try
            {
                var releaseType = ParseReleaseType();

                if (
                    ConsoleHelper.PromptUser("Register new release? " + releaseType.ToString() + " = " + ApplicationVersion,
                        new ConsoleKey[] { ConsoleKey.Y, ConsoleKey.N }) == ConsoleKey.N)
                {
                    return;
                }
                DeployHelper.RegisterClientVersion(CentralServiceApiKey, AppDisplayName, releaseType, ServiceClients.Contracts.DevicePlatform.Windows, ApplicationVersion, DateTime.Now);
            }
            catch (Exception ex)
            {
                Console.ForegroundColor = ConsoleColor.Red;
                Console.WriteLine("Failed: " + ex.Message);
                Console.WriteLine(ex.StackTrace);
                if (ex.InnerException != null)
                {
                    Console.WriteLine("Inner Exception:");
                    Console.WriteLine("Failed: " + ex.InnerException.Message);
                    Console.WriteLine(ex.InnerException.StackTrace);
                }
                Console.ResetColor();

                if (
                    ConsoleHelper.PromptUser("Register Client Version failed. Try again?",
                        new[] { ConsoleKey.Y, ConsoleKey.N }) == ConsoleKey.N)
                {

                    if (
                        ConsoleHelper.PromptUser("Continue deploy?",
                            new ConsoleKey[] { ConsoleKey.Y, ConsoleKey.N }) == ConsoleKey.N)
                    {
                        throw new Exception("Deploy cancelled!");
                    }
                }
                else
                {
                    RegisterClientVersion();
                }
            }
        }

        public void TagClientVersion()
        {
            try
            {
                var releaseType = ParseReleaseType();
                if (!ConsoleHelper.PromptUserToConfirm($"Tag new release? {releaseType} = {ApplicationVersion}"))
                {
                    return;
                }

                var solutionFileInfo = new FileInfo(SolutionPath);
                if (solutionFileInfo.Directory == null)
                {
                    throw new Exception($"SolutionPath '{SolutionPath}' was invalid.");
                }

                var desktopRepoPath = solutionFileInfo.Directory.FullName;
                var lagunaRepoPath = Path.Combine(Directory.GetParent(desktopRepoPath).FullName, "laguna");
                
                TagRepo(desktopRepoPath, ApplicationVersion);
                TagRepo(lagunaRepoPath, ApplicationVersion);
            }
            catch (Exception ex)
            {
                Console.ForegroundColor = ConsoleColor.Red;
                Console.WriteLine("Failed to tag new release: " + ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
        }

        private void TagRepo(string repoLocation, string applicationVersion)
        {
            try
            {
                DeployHelper.TagRepo(repoLocation, applicationVersion);
            }
            catch (Exception ex)
            {
                Console.ForegroundColor = ConsoleColor.Yellow;
                Console.WriteLine($"Failed to tag new release: {ex.Message}");
                Console.WriteLine(ex.StackTrace);
            }
        }

        private ServiceClients.Contracts.ReleaseType ParseReleaseType()
        {
            return (ServiceClients.Contracts.ReleaseType)Enum.Parse(typeof(ServiceClients.Contracts.ReleaseType), ReleaseType, true);
        }

        private static string GetTopLevelDomain(string url)
        {
            var uri = new Uri(url);
            var host = uri.Host;
            var hostParts = host.Split('.');
            if (hostParts.Length < 2)
            {
                throw new Exception("Invalid domain: " + host);
            }
            return hostParts[hostParts.Length - 2] + "." + hostParts[hostParts.Length - 1]; 
        }

        private static string GetRelativePart(string url)
        {
            var uri = new Uri(url);
            return uri.PathAndQuery;            
        }
        

    }
}
