﻿using System;
using System.Diagnostics;
using System.IO;
using Curse.ServiceClients;

namespace Curse.Voice.UpdateDeployment
{
    public interface IDeployHelper
    {
        void TagRepo(string repoLocation, string applicationVersion);
        void RegisterClientVersion(string centralServiceApiKey, string appDisplayName, ServiceClients.Contracts.ReleaseType releaseType, ServiceClients.Contracts.DevicePlatform platform, string applicationVersion, DateTime time);
    }

    public class DeployHelper : IDeployHelper
    {
        private readonly MongooseClient _mongooseClient;

        public DeployHelper(MongooseClient mongooseClient)
        {
            _mongooseClient = mongooseClient;
        }

        public void TagRepo(string repoPath, string applicationVersion)
        {
            if (!Directory.Exists(repoPath))
            {
                throw new Exception($"Directory '{repoPath} did not exist.");
            }

            var tag = $"Windows/v{applicationVersion}";
            var description = $"Windows/v{applicationVersion} deployed {DateTimeExtensions.GetPacificTimeNow().ToString("dddd, M-d-yyyy @ h:mm:ss")} PST";

            Console.WriteLine($"Tagging client version {tag} - {description}...");

            var processStartInfo = new ProcessStartInfo
            {
                FileName = "git",
                Arguments = $"tag -a {tag} -m \"{description}\"",
                UseShellExecute = false,
                RedirectStandardOutput = true,
                RedirectStandardError = true,
                CreateNoWindow = true,
                WorkingDirectory = repoPath
            };
            var process = new Process { StartInfo = processStartInfo };
            if (!process.Start())
            {
                throw new Exception("git process failed to start.  Could not find git.exe in the PATH.");
            }
            Console.WriteLine(process.StandardOutput.ReadToEnd());
            Console.WriteLine(process.StandardError.ReadToEnd());
            if (!process.WaitForExit(5000))
            {
                throw new Exception($"Failed to tag client version for '{repoPath}'");
            }
            if (process.ExitCode != 0)
            {
                throw new Exception($"Failed to tag client version for '{repoPath}', exit code {process.ExitCode}");
            }

            Console.WriteLine($"Tagged client version {tag} - {description}.");

            PushTag(repoPath, tag);

            Console.WriteLine($"Pushed tag for client version {tag} - {description}.");
        }

        private static void PushTag(string repoPath, string tag)
        {
            var processStartInfo = new ProcessStartInfo
            {
                FileName = "git",
                Arguments = $"push origin \"{tag}\"",
                UseShellExecute = false,
                RedirectStandardOutput = true,
                RedirectStandardError = true,
                CreateNoWindow = true,
                WorkingDirectory = repoPath
            };
            var process = new Process { StartInfo = processStartInfo };
            if (!process.Start())
            {
                throw new Exception("git process failed to start.  Could not find git.exe in the PATH.");
            }
            Console.WriteLine(process.StandardOutput.ReadToEnd());
            Console.WriteLine(process.StandardError.ReadToEnd());
            if (!process.WaitForExit(5000))
            {
                throw new Exception($"Failed to push tag for '{repoPath}' and tag '{tag}'");
            }
            if (process.ExitCode != 0)
            {
                throw new Exception($"Failed to push tag version for '{repoPath}' and tag '{tag}', exit code {process.ExitCode}");
            }
        }

        public void RegisterClientVersion(string centralServiceApiKey, string appDisplayName, ServiceClients.Contracts.ReleaseType releaseType, ServiceClients.Contracts.DevicePlatform platform, string applicationVersion, DateTime time)
        {
            Console.WriteLine("Registering client version with Mongoose...");
            _mongooseClient.PublishVersion(new MongooseClient.PublishVersionRequest
            {
                ApplicationName = appDisplayName,
                ReleaseType = releaseType,
                Platform = platform,
                ReleaseDate = time,
                Version = applicationVersion,
            });

            Console.WriteLine("Registering client version with Vox...");
            Services.Instance.ClientVersion.RegisterClientVersion(new ServiceClients.Contracts.RegisterVersionRequest
            {
                ApiKey = centralServiceApiKey,
                ReleaseType = releaseType,
                VersionString = applicationVersion,
                Platform = platform,
            });

            Console.WriteLine("RegisterClientVersion complete.");
        }
    }
}
