﻿using System.Data.Odbc;
using System.Diagnostics;
using System.Net;
using System.ServiceModel.Description;
using System.Web.Script.Serialization;
using Curse.Voice.Helpers;
using Curse.Voice.UpdateDeployment.HostUpdateService;
using ICSharpCode.SharpZipLib.Zip;
using Microsoft.Build.Framework;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.ServiceModel;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using Curse.VaultSharp;

namespace Curse.Voice.UpdateDeployment
{
    public class ServerUpdateDeploymentTask : ITask, ICancelableTask
    {
        public IBuildEngine BuildEngine { get; set; }
        public ITaskHost HostObject { get; set; }

        public string SourceFolder { get; set; }
        public string CentralServiceUrl { get; set; }
        public string CentralServiceApiKey { get; set; }
        public string ServerVersion { get; private set; }
        public string MinimumClientVersion { get; set; }
        public string Environment { get; set; }

        public bool AutoCommit { get; set; }

        public int JobID { get; private set; }
        private bool _isCancelled = false;
        private HostUpdateServiceClient _client; 

        public void Cancel()
        {
            _isCancelled = true;

            if (JobID > 0)
            {
                using (HostUpdateServiceClient client = GetClient())
                {
                    client.CancelUpdate(CentralServiceApiKey, JobID);
                }
            }
        }

        private HostUpdateServiceClient GetClient()
        {
            if(_client != null && _client.State == CommunicationState.Opened)
            {
                return _client;
            }
            var binding = BindingHelper.GetWsHttpBinding(30);            
            binding.Security.Mode = SecurityMode.None;
            binding.SendTimeout = TimeSpan.FromMinutes(120);
            binding.ReceiveTimeout = TimeSpan.FromMinutes(120);
            _client = new HostUpdateServiceClient(binding, new EndpointAddress(CentralServiceUrl));
            return _client;
        }

        private static string GetVaultEnvironment(VoiceHostEnvironment environment)
        {
            switch (environment)
            {
                case VoiceHostEnvironment.Alpha:
                case VoiceHostEnvironment.Beta:
                case VoiceHostEnvironment.Release:
                    return "release";
                case VoiceHostEnvironment.Staging:
                    return "staging";
                default:
                    Console.WriteLine("Could not find a vault environment for " + environment);
                    return null;
            }
        }

        public bool Execute()
        {

            Console.WriteLine("------------------------------------------------------------");

            VoiceHostEnvironment env;
            if (!Enum.TryParse(Environment, out env))
            {
                Console.WriteLine("Unable to parse environment from string '" + Environment + "'");
                return false;
            }

            var vaultEnvironment = GetVaultEnvironment(env);
            if (string.IsNullOrEmpty(vaultEnvironment))
            {
                Console.WriteLine("Unable to get the Central API Key for environment: " + Environment);
                return false;
            }

            Console.WriteLine("Pulling Central Service API Key from the Vault");
            CentralServiceApiKey = VaultSharpHelper.GetSecretValue("api-keys/central", vaultEnvironment) as string;
            if (string.IsNullOrEmpty(CentralServiceApiKey))
            {
                Console.WriteLine("Unable to get the Central API Key for environment: " + Environment);
                return false;
            }

            bool useExistingVersion = false;
            if (!AutoCommit)
            {
                useExistingVersion = ConsoleHelper.PromptUser("Would you like to deploy an existing version, insead of the one that was just compiled?", new[] {ConsoleKey.Y, ConsoleKey.N,}) ==
                                         ConsoleKey.Y;
            }

            var serverVersionID = 0;
            if (useExistingVersion)
            {
                Console.WriteLine("Please enter the ID of an existing server version.");
                var userInput = Console.ReadLine();
                if (!int.TryParse(userInput, out serverVersionID))
                {
                    Console.WriteLine("Unable to parse version ID from string '" + userInput + "'");
                    return false;
                }
            }

            MinimumClientVersion = string.Empty;
            if (!useExistingVersion && !AutoCommit)
            {
                Console.WriteLine("Does this server deploy require a minimum client version? If so, enter it now, or simply press enter to continue.");
                string minClientVersion = Console.ReadLine();

                if (!string.IsNullOrEmpty(minClientVersion))
                {
                    MinimumClientVersion = Version.Parse(minClientVersion).ToString();
                }
            }

            var disableRollout = false;
            var hostID = 0;
            var regionID = 0;
            var hostModes = 0;

            if (!AutoCommit)
            {
                var rolloutMode = ConsoleHelper.PromptUser("Do you want to roll this out to a specific Host(H), Region(R) or All(A)?", new[] {ConsoleKey.H, ConsoleKey.R, ConsoleKey.A});

                if (rolloutMode == ConsoleKey.H)
                {
                    disableRollout = true;

                    Console.WriteLine("Please enter a Host ID to use as the release candidate target.");
                    var userInput = Console.ReadLine();
                    if (!int.TryParse(userInput, out hostID))
                    {
                        Console.WriteLine("Unable to parse Host ID from string '" + userInput + "'");
                        return false;
                    }

                }
                else if (rolloutMode == ConsoleKey.R)
                {
                    disableRollout = true;
                    Console.WriteLine("Please enter a Region ID to use as the release candidate target.");
                    var userInput = Console.ReadLine();
                    if (!int.TryParse(userInput, out regionID))
                    {
                        Console.WriteLine("Unable to parse Region ID from string '" + userInput + "'");
                        return false;
                    }
                }

                if (hostID == 0)
                {
                    var hostMode = ConsoleHelper.PromptUser("Do you want to roll this out to Audio(A), Video(V) or All(L) hosts?", new[] { ConsoleKey.A, ConsoleKey.V, ConsoleKey.L });
                    switch (hostMode)
                    {
                        case ConsoleKey.A:
                            hostModes = 1;
                            break;

                        case ConsoleKey.V:
                            hostModes = 2;
                            break;                        
                        case ConsoleKey.L:
                            hostModes = 0;
                            break;                            
                        default:
                            Console.WriteLine("Unknown host mode");
                            return false;
                    }

                    if (hostModes != 0)
                    {
                        disableRollout = true;
                    }
                }
            }

            var abortOnFailure = true;
            if (hostID != 0)
            {
                abortOnFailure = false; // No need to abort on a single-host deploy
            }
            else if (!AutoCommit)
            {
                abortOnFailure = ConsoleHelper.PromptUser("If a host fails to update, do you want to abort the rollout?", new[] { ConsoleKey.Y, ConsoleKey.N, }) == ConsoleKey.Y;
            }

            var criticality = 4; // "Low chance of impact" (TODO: prompt for this?)
            var desc = "";
            var target = "";

            // Echo settings and prompt for final confirmation
            Console.WriteLine();
            Console.WriteLine();
            Console.WriteLine("Environment: " + env);
            if (useExistingVersion)
            {
                criticality = 5; // "Extremely low/nil chance of impact" (pre-existing version)
                desc = "deploy v" + serverVersionID;
                Console.WriteLine("Version: " + serverVersionID);
            }
            else
            {
                desc = "deploy new (min " + MinimumClientVersion + ")";
                Console.WriteLine("MinClientVersion: " + MinimumClientVersion);
            }

            if (!disableRollout)
            {
                target = "ALL hosts in ALL regions";
                Console.ForegroundColor = ConsoleColor.Red;
                Console.WriteLine("WARNING: This update will be deployed to ALL voice hosts in ALL regions");
                Console.ResetColor();
            }
            else if (hostID != 0)
            {
                target = "host " + hostID;
                Console.WriteLine("HostID: " + hostID);
            }
            else
            {
                target = "region " + regionID + "; types " + hostModes;
                Console.WriteLine("RegionID: " + regionID);
                Console.WriteLine("HostModes: " + hostModes);
            }

            desc += "; " + target;
            if (abortOnFailure)
            {
                desc += "; abortOnFailure";
            }

            if (ConsoleHelper.PromptUser("Are you sure you want to continue?", new[] { ConsoleKey.Y, ConsoleKey.N, }) == ConsoleKey.N)
            {
                Console.WriteLine("Aborting rollout");
                return false;
            }

            // Report deploy to Changelog
            if (env == VoiceHostEnvironment.Beta || env == VoiceHostEnvironment.Release)
            {
                try
                {
                    var request = (HttpWebRequest)WebRequest.Create("https://changelog.internal.justin.tv/api/events");
                    request.ContentType = "application/json";
                    request.Method = "POST";

                    using (var streamWriter = new StreamWriter(request.GetRequestStream()))
                    {
                        var json = new JavaScriptSerializer().Serialize(new
                        {
                            criticality = criticality,
                            unix_timestamp = (int)(DateTime.UtcNow.Subtract(new DateTime(1970, 1, 1))).TotalSeconds,
                            category = "voice",
                            command = System.Environment.CommandLine,
                            description = desc,
                            // optional
                            additional_data = ConsoleHelper.PromptUser("Description: "),
                            username = System.Environment.UserName,
                            source = System.Environment.MachineName,
                            target = target,
                        });

                        streamWriter.Write(json);
                    }

                    var response = (HttpWebResponse)request.GetResponse();
                    if (response.StatusCode != HttpStatusCode.OK && response.StatusCode != HttpStatusCode.Created)
                    {
                        Console.WriteLine("Changelog update failed: " + (int)response.StatusCode);
                        using (var streamReader = new StreamReader(response.GetResponseStream()))
                        {
                            Console.WriteLine(streamReader.ReadToEnd());
                        }
                        return false;
                    }
                }
                catch (Exception ex)
                {
                    Console.WriteLine(ex);
                    Console.WriteLine("Failed to post to Changelog, aborting rollout");
                    return false;
                }
            }

            // Trigger the deploy
            bool success = false;
            if (useExistingVersion)
            {
                success = Deploy(abortOnFailure, disableRollout, hostID, regionID, hostModes, env, serverVersionID);
            }
            else
            {
                success = UploadAndDeploy(abortOnFailure, disableRollout, hostID, regionID, hostModes, env);
            }

            if (success && JobID > 0)
            {
                Task.Factory.StartNew(() =>
                {                
                    using (var client = GetClient())
                    {
                        var currentStringLength = 0;
                        VoiceHostDeploymentStatus deploymentStatus = null;
                        Console.WriteLine();
                        do
                        {
                            deploymentStatus = client.GetDeploymentStatus(CentralServiceApiKey, JobID);
                            if (deploymentStatus != null)
                            {
                                if (!string.IsNullOrWhiteSpace(deploymentStatus.MessageLog))
                                {
                                    string msg = deploymentStatus.MessageLog.Substring(currentStringLength);
                                    
                                    if(!string.IsNullOrWhiteSpace(msg))
                                    {
                                        currentStringLength = deploymentStatus.MessageLog.Length;
                                        Console.WriteLine(msg);
                                    }
                                    
                                }
                                Task.Delay(400).Wait();                          
                            }
                        }
                        while (deploymentStatus == null || deploymentStatus.DeploymentComplete == false);
                    }
                }).Wait();             
            }

            return success;            
        }

        private bool Deploy(bool abortOnFailure, bool disableRollout, int hostID, int regionID, int hostModes, VoiceHostEnvironment environment, int serverVersionID)
        {
            bool successful = false;
            using (var client = GetClient())
            {                
                Console.Write("Uploading payload to service...");                
                try
                {                    
                    var jobID = 0;
                    string message = null;
                    successful = client.DeployUpdate(abortOnFailure, CentralServiceApiKey, disableRollout, environment, hostID, hostModes, regionID, serverVersionID, out jobID, out message);                    
                    JobID = jobID;                    
                    if(!string.IsNullOrWhiteSpace(message))
                    {
                        Console.WriteLine(message);
                    }
                }
                catch (Exception ex)
                {
                    Console.WriteLine("Failed! Error: " + ex.Message);
                    successful = false;
                }                              
            }

            return successful;
        }

        private bool UploadAndDeploy(bool abortOnFailure, bool disableRollout, int hostID, int regionID, int hostModes, VoiceHostEnvironment environment)
        {
            var fileVersionInfo = FileVersionInfo.GetVersionInfo(Path.Combine(SourceFolder, "CurseVoice.exe"));
            ServerVersion = fileVersionInfo.ProductVersion;


            Console.WriteLine("Starting deployment of server version " + ServerVersion + " for the " + environment + " environment.");

            Console.WriteLine("Replacing secrets");
            var vaultEnvironment = GetVaultEnvironment(environment);
            if (string.IsNullOrEmpty(vaultEnvironment))
            {
                Console.WriteLine("Failed!");
                return false;
            }

            try
            {
                VaultSharpHelper.ReplaceSecrets(SourceFolder, vaultEnvironment);
            }
            catch (Exception ex)
            {
                Console.WriteLine("Failed! Error: " + ex.Message);
                return false;
            }

            Console.WriteLine("Done!");

            var outputFile = Path.GetTempFileName();
            var fastZip = new FastZip();
            Console.Write("Creating zip file...");

            try
            {
                fastZip.CreateZip(outputFile, SourceFolder, true, @"\.dll$;\.pem$;\.exe$;\.pdb$;\." + Environment + @".config$;\.exe.config$", "Configuration;pem;x64");
            }
            catch (Exception ex)
            {
                Console.WriteLine("Failed! Error: " + ex.Message);
                return false;
            }

            Console.WriteLine("Done!");

            bool successful = false;
            using (var client = GetClient())
            {

                using (var stream = new FileStream(outputFile, FileMode.Open, FileAccess.Read))
                {
                    Console.Write("Uploading payload to service...");                    

                    try
                    {
                        var jobID = 0;
                        string message = null;
                        successful = client.UploadUpdate(abortOnFailure, CentralServiceApiKey, disableRollout, environment, hostID, hostModes, MinimumClientVersion, regionID, ServerVersion, stream, out jobID, out message);                        
                        JobID = jobID;                        

                        if(!string.IsNullOrWhiteSpace(message))
                        {
                            Console.WriteLine(message);
                        }
                    }
                    catch (Exception ex)
                    {
                        Console.WriteLine("Failed! Error: " + ex.Message);
                        successful = false;
                    }               
                }                
            }

            return successful;
        }
    }


}
