﻿using System;
using System.ServiceModel;
using System.ServiceProcess;
using Curse.Logging;
using Curse.Voice.UpdateManagement;
using System.IO;
using Curse.Voice.UpdateService.Configuration;
using System.Diagnostics;
using System.Threading;
using ICSharpCode.SharpZipLib.Zip;

namespace Curse.Voice.UpdateService
{

    [ServiceBehavior(AddressFilterMode = AddressFilterMode.Any, ConcurrencyMode = ConcurrencyMode.Multiple, InstanceContextMode = InstanceContextMode.Single, UseSynchronizationContext = false)]
    public class VoiceHostUpdateListener : IVoiceHostUpdateListener
    {

        private static bool _isUpdating = false;

        public bool PushUpdate(string apiKey, Version newVersion, byte[] payload)
        {
            if (_isUpdating)
            {
                return false;
            }

            Logger.Info("Received update for version " + newVersion);
            _isUpdating = true;

            try
            {
                var currentVersionFolder = GetCurrentInstallFolder();
                var newVersionFolder = GetVersionedFolder(newVersion);
                var tempFilePath = Path.GetTempFileName();

                // Extract the install to a versioned folder
                Logger.Info("Saving Payload to '" + tempFilePath + "'");
                File.WriteAllBytes(tempFilePath, payload);

                // Extract the install to a versioned folder
                Logger.Info("Extracting file from '" + tempFilePath + "' to '" + newVersionFolder.FullName + "'");

                if (!Extract(newVersionFolder, tempFilePath))
                {
                    return false;
                }

                // Next stop the current service
                Logger.Info("Stopping Service");
                if (!StopService())
                {
                    return false;
                }

                

                // Read in the version info from the current install
                Version lastVersion = null;
                Logger.Info("Getting Current Runtime Version");
                if (!GetAssemblyVersion(currentVersionFolder, out lastVersion))
                {
                    lastVersion = new Version("1.0.0.0");
                }

                var lastVersionFolder = GetVersionedFolder(lastVersion);

                if (!lastVersionFolder.Exists)
                {
                    lastVersionFolder.Create();
                }

                if (currentVersionFolder.Exists)
                {
                    if (!newVersion.Equals(lastVersion))
                    {
                        Logger.Info(string.Format("Copying '{0}' to '{1}'", currentVersionFolder.FullName, lastVersionFolder.FullName));
                        CopyFolderContents(currentVersionFolder.FullName, lastVersionFolder.FullName);
                    }

                    Logger.Info("Deleting contents of '{0}'", currentVersionFolder.FullName);
                    if (!DeleteFolderContents(currentVersionFolder.FullName, true))
                    {
                        return false;
                    }

                }


                Logger.Info(string.Format("Copying '{0}' to '{1}'", newVersionFolder.FullName, currentVersionFolder.FullName));
                if (!CopyFolderContents(newVersionFolder.FullName, currentVersionFolder.FullName))
                {
                    return false;
                }

                // Finally, start the service
                Logger.Info("Starting Service");
                if (!StartService())
                {
                    return false;
                }

                Logger.Info("Update Successful");
                return true;

            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process update!");

                return false;
            }
            finally
            {
                _isUpdating = false;
            }
            
           
        }

        private static bool DeleteFolderContents(string sourcePath, bool isRootFolder = false)
        {
            sourcePath = sourcePath.EndsWith(@"\") ? sourcePath : sourcePath + @"\";

            for (var i = 0; i < 3; i++)
            {

                try
                {
                    if (Directory.Exists(sourcePath))
                    {

                        foreach (var file in Directory.GetFiles(sourcePath, "*", SearchOption.AllDirectories))
                        {
                            Logger.Info("Deleting file '" + file + "'...");
                            File.Delete(file);
                        }

                        foreach (var drs in Directory.GetDirectories(sourcePath, "*", SearchOption.TopDirectoryOnly))
                        {
                            if (!DeleteFolderContents(drs))
                            {
                                Logger.Error("Failed to delete nested folder '" + drs + "'");
                                return false;
                            }
                        }

                        if (!isRootFolder)
                        {
                            Logger.Info("Deleting folder '" + sourcePath + "'...");
                            Directory.Delete(sourcePath);
                        }
                    }
                    return true;
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to delete folder contents of '" + sourcePath + "'");
                    
                }

                Logger.Info("Failed to delete folder, trying again in 1 second...");

                Thread.Sleep(1000);
            }

            return false;
        }


        private static bool CopyFolderContents(string sourcePath, string destinationPath)
        {
            sourcePath = sourcePath.EndsWith(@"\") ? sourcePath : sourcePath + @"\";
            destinationPath = destinationPath.EndsWith(@"\") ? destinationPath : destinationPath + @"\";

            try
            {
                if (Directory.Exists(sourcePath))
                {
                    if (Directory.Exists(destinationPath) == false)
                    {
                        Directory.CreateDirectory(destinationPath);
                    }

                    foreach (string files in Directory.GetFiles(sourcePath))
                    {
                        var fileInfo = new FileInfo(files);
                        fileInfo.CopyTo(string.Format(@"{0}\{1}", destinationPath, fileInfo.Name), true);
                    }

                    foreach (string drs in Directory.GetDirectories(sourcePath))
                    {
                        var directoryInfo = new DirectoryInfo(drs);
                        if (CopyFolderContents(drs, destinationPath + directoryInfo.Name) == false)
                        {
                            return false;
                        }
                    }
                }
                return true;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to copy folder contents from '" + sourcePath + "' to '" + destinationPath + "'");
                return false;
            }
        }


        private static bool Extract(DirectoryInfo destination, string sourceFilePath)
        {
            try
            {
                // Ensure we can create a folder in the install location                
                if (!destination.Exists)
                {
                    destination.Create();
                }

                // First, extract the uploaded zip to a local temp file
                
                var fastZip = new FastZip();
                fastZip.ExtractZip(sourceFilePath, destination.FullName, FastZip.Overwrite.Always, null, null, null, true);                
                
                return true;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to extract payload");
                return false;
            }
        }

        private static DirectoryInfo GetVersionedFolder(Version version)
        {
            return new DirectoryInfo(Path.Combine(UpdateServiceConfiguration.Instance.LocalInstallPath, version.ToString().Replace(".", "-")));
        }

        private static DirectoryInfo GetCurrentInstallFolder()
        {
            return new DirectoryInfo(Path.Combine(UpdateServiceConfiguration.Instance.LocalInstallPath, "Current"));
        }

        private static bool GetAssemblyVersion(DirectoryInfo folder, out Version version)
        {
            try
            {

                string currentExe = Path.Combine(folder.FullName, "CurseVoice.exe");
                FileVersionInfo fileVersionInfo = FileVersionInfo.GetVersionInfo(currentExe);                
                version = Version.Parse(fileVersionInfo.ProductVersion);
                return true;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to determine the current version of CurseVoice service");
                version = null;
                return false;
            }

        }

        private static bool StartService()
        {
            try
            {
                // Next stop the current service
                using (ServiceController sc = new ServiceController("CurseVoice"))
                {                    
                    sc.Start();
                    sc.Refresh();

                    int i = 0;
                    while (sc.Status != ServiceControllerStatus.Running)
                    {
                        if (++i > 100 || sc.Status == ServiceControllerStatus.Stopped)
                        {
                            Logger.Error("Failed to start service!");
                            return false;
                        }
                        Thread.Sleep(100);
                        sc.Refresh();

                    }
                }                
                return true;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to stop CurseVoice service");
                return false;
            }

        }


        private static bool StopService()
        {
            try
            {
                // Next stop the current service
                using (var sc = new ServiceController("CurseVoice"))
                {
                    if (sc.Status == ServiceControllerStatus.Stopped)
                    {
                        return true;
                    }

                    sc.Stop();                   

                    int i = 0;
                    while (sc.Status != ServiceControllerStatus.Stopped)
                    {                        
                        if (++i > 30)
                        {
                            Logger.Error("Failed to stop service, after 30 atempts!");
                            return false;
                        }
                        Thread.Sleep(1000);
                        sc.Refresh();
                    }
                }
                return true;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to stop CurseVoice service");
                return false;
            }

        }

    }
}
