﻿using System;
using System.Net;
using Curse.Logging;
using Curse.SocketInterface;
using Curse.WebRTC.OpenSSL;
using System.Reflection;
using System.IO;

namespace Curse.WebRTC
{
    internal class Dtls : IDisposable
    {
        private static readonly LogCategory Logger = new LogCategory("Dtls")
            { Throttle = TimeSpan.FromSeconds(30) };

        #region Context and Certificate

        private static SslContext _ctx;
        private static SslPrivateKey _key;
        private static SslCertificate _cert;

        public static void Init()
        {
            if (_ctx != null)
                return;

            _ctx = SslContext.CreateDtlsServer();

            _ctx.SetCipherList("DEFAULT:!NULL:!aNULL:!SHA256:!SHA384:!aECDH:!AESGCM+AES256:!aPSK");

            Native.SSL_CTX_set_verify(_ctx, Native.SSL_VERIFY_PEER, VerifyPeer);

            _ctx.SetTslExtUseSrtp("SRTP_AES128_CM_SHA1_32");

            //_key = SslPrivateKey.GenerateElipticCurveKey();
            _key = SslPrivateKey.GenerateRsaKey();
            _ctx.Use(_key);

            _cert = SslCertificate.Generate(_key);
            _ctx.Use(_cert);
            Fingerprint = _cert.Digest();

            _ctx.CheckPrivateKey();

            //Native.SSL_CTX_set_options(_ctx, Native.SSL_OP_SINGLE_DH_USE);

            Native.SSL_CTX_set_tmp_dh_callback(_ctx, TmpDHCallback);
        }

        private static readonly Native.VerifyCertCallback VerifyPeer = _VerifyPeer;

        private static int _VerifyPeer(int ok, IntPtr x509_store_ctx)
        {
            return 1;
        }

        public static string Fingerprint { get; private set; }

        #endregion

        private readonly Ssl _ssl;
        private readonly Bio _bioIn;
        private readonly Bio _bioOut;

        private bool _isDisposed;

        public Dtls()
        {
            _ctx.SetOptions(Native.SSL_OP_NO_QUERY_MTU);
            _ssl = Ssl.Create(_ctx);
            _ssl.DtlsSetLinkMtu(SocketConstants.BufferSize);

            _bioIn = Bio.CreateMemory();
            _bioOut = Bio.CreateMemory();

            _ssl.SetBio(ref _bioIn, ref _bioOut);

            _ssl.SetAcceptState();
        }

        public void Dispose()
        {
            lock (_ssl)
            {
                _isDisposed = true;

                // No need to dispose _bioIn/_bioOut since they don't own their handles
                // It's important that they are never used after _ssl is disposed, though
                _ssl.Dispose();
            }
        }

        public bool IsConnected
        {
            get
            {
                lock (_ssl)
                {
                    return !_isDisposed && _ssl.State == SslState.Ok;
                }
            }
        }

        public void HandleTraffic(RTCChannelProcessor processor, SocketBuffer msg)
        {
            lock (_ssl)
            {
                if (_isDisposed)
                    throw new ObjectDisposedException("DTLS");

                // TODO: Figure out how to use HelloVerifyRequest to prevent IP spoofing attacks?

                var written = _bioIn.Write(msg.Buffer, msg.Offset, msg.Count);
                if (written != msg.Count)
                {
                    Logger.Warn("Failed to write full message", new {msg.Count, Written = written});
                }

                if (!IsConnected)
                {
                    _ssl.DoHandshake();
                }

                if (_bioOut.Pending > 0)
                {
                    var buf = new byte[SocketConstants.BufferSize];
                    var length = _bioOut.Read(buf, 0, buf.Length);
                    processor.StartSend(msg.Sender, new ByteBuffer(buf, 0, length));
                }
            }
        }

        public void CreateSrtp(out Srtp srtpIn, out Srtp srtpOut)
        {
            lock (_ssl)
            {
                if (_isDisposed)
                    throw new ObjectDisposedException("DTLS");

                Srtp.Create(_ssl, out srtpIn, out srtpOut);
            }
        }

        #region Diffie-Hellman Parameters

        private static readonly Native.tmp_dh_callback TmpDHCallback = _TmpDHCallback;

        private static IntPtr _TmpDHCallback(IntPtr ssl, int isExport, int keyLength)
        {
            switch (keyLength)
            {
                case 512:
                    return DH512;

                case 1024:
                    return DH1024;

                case 2048:
                    return DH2048;

                case 4096:
                    return DH4096;
            }
            return IntPtr.Zero;
        }

        private static readonly IntPtr DH512 = LoadDHparams("pem\\dh512.pem");
        private static readonly IntPtr DH1024 = LoadDHparams("pem\\dh1024.pem");
        private static readonly IntPtr DH2048 = LoadDHparams("pem\\dh2048.pem");
        private static readonly IntPtr DH4096 = LoadDHparams("pem\\dh4096.pem");

        private static IntPtr LoadDHparams(string filename)
        {
            // Get full pem file path
            var fullPath = Path.Combine(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location), filename);

            if (!File.Exists(fullPath))
            {
                throw new IOException("Unable to find pem file at " + fullPath);
            }

            using (var bio = Bio.CreateFile(fullPath))
            {
                return Native.PEM_read_bio_DHparams(bio, IntPtr.Zero, null, IntPtr.Zero);
            }
        }

        #endregion
    }
}
