﻿using System;
using System.Runtime.ConstrainedExecution;
using System.Runtime.InteropServices;

namespace Curse.WebRTC.OpenSSL
{
    internal class Bio : SslSafeHandle
    {
        #region SafeHandle

        [ReliabilityContract(Consistency.WillNotCorruptState, Cer.MayFail)]
        private Bio()
        {
        }

        [ReliabilityContract(Consistency.WillNotCorruptState, Cer.MayFail)]
        private Bio(bool ownsHandle)
            : base(ownsHandle)
        {
        }

        [ReliabilityContract(Consistency.WillNotCorruptState, Cer.Success)]
        protected override bool ReleaseHandle()
        {
            Native.BIO_free(handle);
            handle = IntPtr.Zero;
            return true;
        }

        #endregion SafeHandle

        public static Bio CreateMemory()
        {
            return Check(Native.BIO_new(Native.BIO_s_mem()));
        }

        public static Bio CreateFile(string filename, string mode = "r")
        {
            return Check(Native.BIO_new_file(filename, mode));
        }

        public Bio ReleaseOwnership()
        {
            SetHandleAsInvalid();
            GC.SuppressFinalize(this);
            return new Bio(false) { handle = handle };
        }

        public int Write(byte[] buf, int offset, int len)
        {
            if ((uint)offset > (uint)buf.Length)
                throw new ArgumentOutOfRangeException("offset");
            if (len < 0 || (uint)(offset + len) > (uint)buf.Length)
                throw new ArgumentOutOfRangeException("len");

            var h = GCHandle.Alloc(buf, GCHandleType.Pinned);
            try
            {
                var ptr = h.AddrOfPinnedObject() + offset;
                return Native.BIO_write(this, ptr, len);
            }
            finally
            {
                h.Free();
            }
        }

        public int Read(byte[] buf, int offset, int len)
        {
            if ((uint)offset > (uint)buf.Length)
                throw new ArgumentOutOfRangeException("offset");
            if (len < 0 || (uint)(offset + len) > (uint)buf.Length)
                throw new ArgumentOutOfRangeException("len");

            var h = GCHandle.Alloc(buf, GCHandleType.Pinned);
            try
            {
                var ptr = h.AddrOfPinnedObject() + offset;
                return Native.BIO_read(this, ptr, len);
            }
            finally
            {
                h.Free();
            }
        }

        public uint Pending
        {
            get { return Native.BIO_ctrl_pending(this); }
        }
    }
}
