﻿// Copyright (c) 2006-2007 Frank Laub
// All rights reserved.

// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
// 3. The name of the author may not be used to endorse or promote products
//    derived from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
// IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
// INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
// THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

using System.Text;

namespace Curse.WebRTC.OpenSSL
{
    /// <summary>
    /// This is a struct that contains a uint for the native openssl error code.
    /// It provides helper methods to convert this error code into strings.
    /// </summary>
    public struct OpenSslError
    {
        private readonly uint _err;

        /// <summary>
        /// Constructs an OpenSslError object.
        /// </summary>
        /// <param name="err">The native error code</param>
        public OpenSslError(uint err)
        {
            _err = err;
        }

        /// <summary>
        /// Returns the native error code
        /// </summary>
        public uint ErrorCode
        {
            get { return _err; }
        }

        /// <summary>
        /// Returns the result of ERR_lib_error_string()
        /// </summary>
        public string Library
        {
            get { return Native.PtrToStringAnsi(Native.ERR_lib_error_string(_err), false); }
        }

        /// <summary>
        /// Returns the results of ERR_reason_error_string()
        /// </summary>
        public string Reason
        {
            get { return Native.PtrToStringAnsi(Native.ERR_reason_error_string(_err), false); }
        }

        /// <summary>
        /// Returns the results of ERR_func_error_string()
        /// </summary>
        public string Function
        {
            get { return Native.PtrToStringAnsi(Native.ERR_func_error_string(_err), false); }
        }

        /// <summary>
        /// Returns the results of ERR_error_string_n()
        /// </summary>
        public string Message
        {
            get
            {
                var buf = new byte[1024];

                buf.Initialize();
                Native.ERR_error_string_n(_err, buf, buf.Length);

                int len;
                for (len = 0; len < buf.Length; len++)
                {
                    if (buf[len] == 0)
                        break;
                }

                return Encoding.ASCII.GetString(buf, 0, len);
            }
        }
    }
}