﻿using System;
using System.Runtime.ConstrainedExecution;

namespace Curse.WebRTC.OpenSSL
{
    internal class Ssl : SslSafeHandle
    {
        #region SafeHandle

        [ReliabilityContract(Consistency.WillNotCorruptState, Cer.MayFail)]
        private Ssl()
        {
        }

        [ReliabilityContract(Consistency.WillNotCorruptState, Cer.Success)]
        protected override bool ReleaseHandle()
        {
            Native.SSL_free(handle);
            handle = IntPtr.Zero;
            return true;
        }

        #endregion SafeHandle

        public static Ssl Create(SslContext ctx)
        {
            return Check(Native.SSL_new(ctx));
        }

        public void SetBio(ref Bio read, ref Bio write)
        {
            Native.SSL_set_bio(this, read, write);

            // ssl now owns these
            read = read.ReleaseOwnership();
            write = write.ReleaseOwnership();
        }

        public void SetAcceptState()
        {
            Native.SSL_set_accept_state(this);
        }

        public SslState State { get { return Native.SSL_state(this); } }

        public void DoHandshake()
        {
            Check(Native.SSL_do_handshake(this));
        }

        public int Read(byte[] buf, int len)
        {
            if ((uint)len > (uint)buf.Length)
                throw new ArgumentOutOfRangeException("len");

            return Native.SSL_read(this, buf, len);
        }

        public OpenSslError GetError(int code)
        {
            return new OpenSslError((uint)Native.SSL_get_error(this, code));
        }

        public void ExportKeyingMaterial(byte[] buf, int len, string label)
        {
            if ((uint)len > (uint)buf.Length)
                throw new ArgumentOutOfRangeException("len");

            Check(Native.SSL_export_keying_material(this, buf, len, label, label.Length, IntPtr.Zero, 0, 0));
        }

        public void DtlsSetLinkMtu(int mtu)
        {
            Check(Native.SSL_ctrl(this, Native.DTLS_CTRL_SET_LINK_MTU, mtu, IntPtr.Zero));
        }
    }
}
