﻿using System;
using System.Runtime.ConstrainedExecution;
using System.Text;

namespace Curse.WebRTC.OpenSSL
{
    internal class SslCertificate : SslSafeHandle
    {
        #region SafeHandle

        [ReliabilityContract(Consistency.WillNotCorruptState, Cer.MayFail)]
        private SslCertificate()
        {
        }

        [ReliabilityContract(Consistency.WillNotCorruptState, Cer.Success)]
        protected override bool ReleaseHandle()
        {
            Native.X509_free(handle);
            handle = IntPtr.Zero;
            return true;
        }

        #endregion SafeHandle

        // Random bits for certificate serial number
        private const int SerialRandBits = 64;

        // Certificate validity lifetime
        private static readonly TimeSpan CertificateLifetime = TimeSpan.FromDays(365 * 10);

        // Certificate validity window.
        // This is to compensate for slightly incorrect system clocks.
        private static readonly TimeSpan CertificateWindow = TimeSpan.FromDays(365 * 10);

        // Null-terminated byte[] dummy name for certificate
        private static readonly byte[] CommonName = Encoding.UTF8.GetBytes("WebRTC\0");

        public static SslCertificate Generate(SslPrivateKey pkey)
        {
            // Create a new certificate with the given key
            var cert = Check(Native.X509_new());
            Check(Native.X509_set_pubkey(cert, pkey));

            // Set the serial number to a random value
            var serial = Check(Native.BN_new());
            try
            {
                var asn1 = Native.X509_get_serialNumber(cert);
                Check(asn1);
                Check(Native.BN_pseudo_rand(serial, SerialRandBits, 0, 0));
                Check(Native.BN_to_ASN1_INTEGER(serial, asn1));
            }
            finally
            {
                Native.BN_free(serial);
            }

            // Set version
            Check(Native.X509_set_version(cert, 0));

            // Set the certificate's subject and issuer to a dummy string
            var name = Check(Native.X509_NAME_new());
            try
            {
                const int NID_commonName = 13;
                const int MBSTRING_UTF8 = 0x1000;

                Check(Native.X509_NAME_add_entry_by_NID(name, NID_commonName, MBSTRING_UTF8, CommonName, -1, -1, 0));
                Check(Native.X509_set_subject_name(cert, name));
                Check(Native.X509_set_issuer_name(cert, name));
            }
            finally
            {
                Native.X509_NAME_free(name);
            }

            // Set valid date range
            var now = DateTime.UtcNow;
            var notBefore = now - CertificateWindow;
            var notAfter = now + CertificateLifetime;

            var time = Check(Native.ASN1_TIME_new());
            try
            {
                var t = (int)notBefore.ToTimeT();
                Check(Native.ASN1_TIME_set(time, notBefore.ToTimeT()));
                Check(Native.X509_set_notBefore(cert, time));
                Check(Native.ASN1_TIME_set(time, notAfter.ToTimeT()));
                Check(Native.X509_set_notAfter(cert, time));
            }
            finally
            {
                Native.ASN1_TIME_free(time);
            }

            // Sign the certificate
            Check(Native.X509_sign(cert, pkey, Native.EVP_sha256()));

            return cert;
        }

        public string Digest()
        {
            uint len;
            var md = new byte[32]; // SHA256

            Check(Native.X509_digest(this, Native.EVP_sha256(), md, out len));
            if (len != md.Length)
            {
                throw new InvalidOperationException("Unexpected digest length, memory corruption may have occurred!");
            }

            const string type = "sha-256 ";
            var sb = new StringBuilder(type.Length + md.Length * 3);
            sb.Append(type);

            foreach (var b in md)
            {
                sb.AppendFormat("{0:X2}:", b);
            }

            // Drop the last ':' character
            sb.Length = sb.Length - 1;

            return sb.ToString();
        }
    }
}
