﻿using System;
using System.Collections.Concurrent;
using System.Net;
using System.Net.Sockets;
using System.Threading;
using Curse.Logging;
using Curse.SocketInterface;

namespace Curse.WebRTC
{
    internal class SendItem
    {
        public Socket Socket;
        public IPAddress Address;
        public int Port;
        public ByteBuffer Data;
    }

    public class RTCChannelProcessor : IDisposable
    {
        private static readonly LogCategory Logger = new LogCategory("RTCChannelProcessor") 
            { Throttle = TimeSpan.FromSeconds(30) };

        private ISocketInterface _socketInterface;
        private readonly ISocketEventArgsPool _socketEventArgsPool;
        private SocketAsyncEventArgs _eventArgs;
        private ConcurrentQueue<SendItem> _messageQueue;
        private readonly object _syncRoot = new object();

        public int ID { get; private set; }

        private volatile bool _isSending = false;
        private volatile bool _isAlive = true;
        private static int _maxQueueSize = 2;

        private static int _instanceCounter = 0;

        public SocketAsyncEventArgs EventArgs
        {
            get { return _eventArgs; }
        }

        public RTCSocketPair Sockets { get; set; }

#if SOCKET_LOGGING

        public void Trace(string message)
        {
            Logger.Trace(message);
        }
#else

        public void Trace(string message)
        {
        }

#endif

        private readonly object _disposeLock = new object();

        ~RTCChannelProcessor()
        {
            Dispose();
        }

        public void Dispose()
        {
            var lockAcquired = Monitor.TryEnter(_disposeLock, 500);

            if (!lockAcquired)
            {
                Logger.Warn("Failed to acquire a lock to dispose the channel processor. A dispose must already be in progress");
                return;
            }

            try
            {
                if (!_isAlive)
                {
                    return;
                }

                _isAlive = false;
                GC.SuppressFinalize(this);

                if (_eventArgs != null)
                {
                    _eventArgs.RemoteEndPoint = null;
                    _eventArgs.UserToken = null;
                    if (_socketEventArgsPool != null)
                    {
                        _socketEventArgsPool.Return(_eventArgs);
                    }
                    else
                    {
                        _eventArgs.Dispose();
                    }

                    _eventArgs = null;
                }

                _socketInterface = null;
                _messageQueue = null;
            }
            finally
            {                
                Monitor.Exit(_disposeLock);                
            }
        }

        public RTCChannelProcessor(ISocketInterface socketInterface, ISocketEventArgsPool socketEventArgsPool, SocketAsyncEventArgs e, RTCSocketPair sockets)
        {
            ID = Interlocked.Increment(ref _instanceCounter);
            if (socketInterface.ClientID > 0)
            {
                ID = socketInterface.ClientID;
            }

            _maxQueueSize = 40;

            //#if DEBUG || LOAD_TESTING
            //            _maxQueueSize = 0;
            //#endif
            _messageQueue = new ConcurrentQueue<SendItem>();
            _socketInterface = socketInterface;
            _socketEventArgsPool = socketEventArgsPool;
            _eventArgs = e;
            _eventArgs.UserToken = this;

            Sockets = sockets;
        }

        public static void Completed(object sender, SocketAsyncEventArgs e)
        {
            // determine which type of operation just completed and call the associated handler
            switch (e.LastOperation)
            {
                case SocketAsyncOperation.SendTo:
                    var channel = (RTCChannelProcessor)e.UserToken;

                    if (channel == null)
                    {
                        // This happens when a connection is disposed while a send operation is pending.
                        // It would be great to prevent this from happening, but because the args pool
                        // uses a queue there's little risk of a re-use before this send has completed.
                        Logger.Debug("Received a UDP completion event after dispose!");
                        return;
                    }

                    try
                    {
                        channel.ProcessSend(e);
                    }
                    catch (Exception ex)
                    {
                        Logger.Error(ex, "Unhandled error during Send operating in UDP_Completed.");
                    }
                    break;

                default:
                    Logger.Warn("Unsupported operation in UDP_Completed", new { e.LastOperation });
                    break;
            }
        }

        public void StartSend(IPEndPoint endPoint, ByteBuffer data, bool rtcp = false)
        {
            // Don't queue up bogus data
            if (endPoint == null || data.Buffer == null || data.Count <= 0)
            {
                return;
            }

            var messageQueue = _messageQueue;
            var eventArgs = _eventArgs;

            if (messageQueue == null || eventArgs == null || !_isAlive) // It's been disposed
            {
                return;
            }

            // If this channel is queued up, discard the data
            if (_maxQueueSize > 0 && messageQueue.Count > _maxQueueSize)
            {
                // This isn't good, but the log message doesn't really help in production.
                Logger.Debug("Channel has reached the maximum queue size!");
                return;
            }

            // Queue up the message
            messageQueue.Enqueue(new SendItem
            {
                Socket = rtcp ? Sockets.RtcpSocket : Sockets.RtpSocket,
                Address = endPoint.Address,
                Port = endPoint.Port,
                Data = data,
            });

            // Try to send it immediately
            StartSend(eventArgs);
        }

        public void StartSend(SocketAsyncEventArgs e)
        {
            try
            {
                // Ensure we're not currently sending, or disposing.
                lock (_syncRoot)
                {
                    if (_isSending || !_isAlive) // Just in case
                    {
                        return;
                    }

                    Trace("StartSend");

                    var messageQueue = _messageQueue;
                    if (messageQueue == null) // It's been disposed
                    {
                        Trace("No current message exists. Exiting events...");
                        return;
                    }

                    SendItem message;
                    if (!messageQueue.TryDequeue(out message))
                    {
                        return;
                    }

                    _isSending = true;
                    try
                    {
                        // Set the destination address and actual length
                        e.SetRemoteEndPoint(message.Address, message.Port);
                        e.SetBufferLength(message.Data.Count);
                        message.Data.BlockCopy(0, e.GetSendBuffer(message.Data.Count), 0, message.Data.Count);

                        // If the send operation did not complete synchronously, return out
                        if (message.Socket.SendToAsync(e))
                        {
                            return; // async callback will handle cleanup
                        }
                    }
                    catch (ObjectDisposedException)
                    {
                        return;
                    }
                }

                ProcessSend(e);
            }
            catch (Exception ex)
            {
                _isSending = false;
                Logger.Error(ex, "Failed to start sending on channel processor.");
            }
        }

        public void ProcessSend(SocketAsyncEventArgs e)
        {
            Trace("ProcessSend");

            lock (_syncRoot)
            {
                // If the socket has been disposed/disconnect, return out
                if (!_isAlive || _socketInterface == null)
                {
                    return;
                }

                // If the send operation was not successful, log the issue, but do nothing else
                if (e.SocketError != SocketError.Success)
                {
                    Logger.Warn("Send failed", new
                    {
                        e.SocketError,
                        e.RemoteEndPoint,
                        e.SendPacketsSendSize,
                        e.SendPacketsFlags,
                        e.Count,
                        e.SocketFlags,
                    });
                    // Don't return here, this might be a transient error
                    // and sending will restart once something new is added
                    // to the queue anyway.
                }

                Trace("Resuming Send");

                // Resume sending
                _isSending = false;
                StartSend(e);
            }
        }
    }
}
