﻿using System;
using System.Collections.Concurrent;
using System.Net;
using System.Net.Sockets;
using System.Threading;
using Curse.Logging;
using Curse.SocketInterface;

namespace Curse.WebRTC
{
    // Based on UdpConnectionManager
    public class RTCSocketPair
    {
        private static readonly LogCategory Logger = new LogCategory("RTCSocketPair")
            { Throttle = TimeSpan.FromSeconds(30) };

        public delegate void ReceiveCallback(RTCSocketPair sockets, SocketBuffer msg, bool rtpc);

        private readonly Socket _rtp;
        private readonly Socket _rtcp;
        private readonly BufferManager _bufferManager;
        private readonly int _bufferSize;
        private readonly ReceiveCallback _recvCallback;
        private readonly object _syncRoot = new object();

        private volatile bool _isFinished;

        public IPAddress Address
        {
            get { return ((IPEndPoint)_rtp.LocalEndPoint).Address; }
        }

        public int RtpPort
        {
            get { return ((IPEndPoint)_rtp.LocalEndPoint).Port; }
        }

        public int RtcpPort
        {
            get { return ((IPEndPoint)_rtcp.LocalEndPoint).Port; }
        }

        public Socket RtpSocket
        {
            get { return _rtp; }
        }

        public Socket RtcpSocket
        {
            get { return _rtcp; }
        }

        public RTCSocketPair(int port, BufferManager bufferManager, int poolSize, ReceiveCallback callback)
            : this(port & ~1, port & ~1 + 1, bufferManager, poolSize, callback)
        {
            // Per the RFC, if only one port is given an adjacent pair is chosen
            // such that the lower port is even (RTP) and the higher port is odd (RTCP)
        }

        public RTCSocketPair(int rtpPort, int rtcpPort, BufferManager bufferManager, int poolSize, ReceiveCallback callback)
        {
            _rtp = BindSocket(rtpPort);
            _rtcp = BindSocket(rtcpPort);
            _bufferManager = bufferManager;
            _bufferSize = bufferManager.BufferSize;
            _recvCallback = callback;

            // Most clients should multiplex over the RTP port,
            // so we only allocate 10% of the args to the RTCP port
            var rtcpPoolSize = Math.Max(1, poolSize / 10);
            var rtpPoolSize = Math.Max(1, poolSize - rtcpPoolSize);

            // TODO: It wouldn't be hard to dynamically update the pool usage
            // TODO: based on actual usage as args are returned

            // Allocate event args and start listening
            for (var i = 0; i < rtpPoolSize; ++i)
            {
                StartReceive(_rtp, CreateArgs());
            }

            for (var i = 0; i < rtcpPoolSize; ++i)
            {
                StartReceive(_rtcp, CreateArgs());
            }
        }

        private static Socket BindSocket(int port)
        {
            var socket = new Socket(AddressFamily.InterNetwork, SocketType.Dgram, ProtocolType.Udp);
            socket.SetSocketOption(SocketOptionLevel.IP, SocketOptionName.PacketInformation, true);
            socket.SetSocketOption(SocketOptionLevel.Socket, SocketOptionName.ReuseAddress, true);
            socket.Blocking = false;
            socket.Bind(new IPEndPoint(IPAddress.Any, port));
            return socket;
        }

        private SocketAsyncEventArgs CreateArgs()
        {
            var e = new SocketAsyncEventArgs();
            e.Completed += Completed;
            e.RemoteEndPoint = new IPEndPoint(IPAddress.Any, 0);
            _bufferManager.SetBuffer(e);
            return e;
        }

        public void Shutdown()
        {
            _isFinished = true;

            lock (_syncRoot)
            {
                CloseSocket(_rtp, "RTP");
                CloseSocket(_rtcp, "RTCP");
            }
        }

        private static void CloseSocket(Socket socket, string name)
        {
            try
            {
                socket.Shutdown(SocketShutdown.Both);
                socket.Close(1);
                socket.Dispose();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to close " + name + " socket");
            }
        }

        private void StartReceive(Socket socket, SocketAsyncEventArgs e)
        {
            lock (_syncRoot)
            {
                e.SetRemoteEndPoint(IPAddress.Any, 0);
                e.SetBufferLength(_bufferSize);

                if (socket.ReceiveFromAsync(e))
                {
                    return; // async callback will handle cleanup
                }
            }

            // Operation completed synchronously so return it to the pool
            OnReceiveComplete(socket, e);
        }

        private void Completed(object sender, SocketAsyncEventArgs e)
        {
            if (_isFinished)
                return;

            switch (e.LastOperation)
            {
                case SocketAsyncOperation.ReceiveFrom:
                    OnReceiveComplete(sender, e);
                    break;

                default:
                    Logger.Warn("Unsupported operation", new { e.LastOperation });
                    break;
            }
        }

        private void OnReceiveComplete(object sender, SocketAsyncEventArgs e)
        {
            var socket = (Socket)sender;
            try
            {
                if (e.SocketError != SocketError.Success)
                    return;

                // Process the packet
                _recvCallback(this, new SocketBuffer(e), socket == _rtcp);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error processing ");
            }
            finally
            {
                // Queue for another receive
                StartReceive(socket, e);
            }
        }
    }
}
