﻿using System;
using System.Net;
using System.Runtime.ConstrainedExecution;
using System.Runtime.InteropServices;
using System.Security;
using System.Security.Cryptography;
using System.Text;
using Curse.Logging;
using Curse.SocketInterface;
using Curse.WebRTC.OpenSSL;

namespace Curse.WebRTC
{
    public class Srtp : SafeHandle
    {
        private const string SRTPDLL = "x64\\srtp.dll";

        private static readonly LogCategory Logger = new LogCategory("Srtp")
            { Throttle = TimeSpan.FromSeconds(30) };

        #region SafeHandle

        [ReliabilityContract(Consistency.WillNotCorruptState, Cer.MayFail)]
        private Srtp()
            : base(IntPtr.Zero, true)
        {
        }

        public override bool IsInvalid
        {
            [System.Security.SecurityCritical]
            get { return handle == IntPtr.Zero; }
        }

        [ReliabilityContract(Consistency.WillNotCorruptState, Cer.MayFail)]
        protected override bool ReleaseHandle()
        {
            if (srtp_dealloc(handle) == err_status_t.err_status_ok)
            {
                handle = IntPtr.Zero;
                return true;
            }
            return false;
        }

        #endregion SafeHandle

        #region Public API

        public static void Init()
        {
            Check(srtp_init());
        }

        internal static void Create(Ssl ssl, out Srtp srtpIn, out Srtp srtpOut)
        {
            // TODO: Client cert validation

            var material = new byte[2 * SRTP_MASTER_LENGTH];
            ssl.ExportKeyingMaterial(material, material.Length, "EXTRACTOR-dtls_srtp");

            const int remoteKey = 0;
            const int localKey = remoteKey + SRTP_MASTER_KEY_LENGTH;
            const int remoteSalt = localKey + SRTP_MASTER_KEY_LENGTH;
            const int localSalt = remoteSalt + SRTP_MASTER_SALT_LENGTH;

            srtpIn = Create(ssrc_type_t.ssrc_any_inbound, material, remoteKey, remoteSalt);
            srtpOut = Create(ssrc_type_t.ssrc_any_outbound, material, localKey, localSalt);
        }

        private const uint WindowSize = 0x2000; // min 64, max 0x7FFF (0x2000 is 256 bytes and ~10 seconds of packets at max video bitrate)
        private const uint WindowSizeWords = (WindowSize + 31)/32;

        private static Srtp Create(ssrc_type_t type, byte[] material, int key, int salt)
        {
            var policy = new srtp_policy_t();
            crypto_policy_set_aes_cm_128_hmac_sha1_32(ref policy.rtp);
            crypto_policy_set_aes_cm_128_hmac_sha1_80(ref policy.rtcp);
            policy.ssrc.type = type;

            policy.key = Marshal.AllocHGlobal(SRTP_MASTER_LENGTH);
            try
            {
                Marshal.Copy(material, key, policy.key, SRTP_MASTER_KEY_LENGTH);
                Marshal.Copy(material, salt, policy.key + SRTP_MASTER_KEY_LENGTH, SRTP_MASTER_SALT_LENGTH);

                policy.window_size = WindowSize;
                policy.allow_repeat_tx = 1;

                Srtp result;
                Check(srtp_create(out result, ref policy));
                return result;
            }
            finally
            {
                Marshal.FreeHGlobal(policy.key);
                policy.key = IntPtr.Zero;
            }
        }

        public void Unprotect(byte[] data, int offset, ref int length)
        {
            if ((uint)offset > (uint)data.Length)
                throw new ArgumentOutOfRangeException("offset");
            if (length < 0 || (uint)(offset + length) > (uint)data.Length)
                throw new ArgumentOutOfRangeException("length");

            var h = GCHandle.Alloc(data, GCHandleType.Pinned);
            try
            {
                var ptr = h.AddrOfPinnedObject() + offset;
                if (((ulong)ptr & 3) != 0)
                    throw new ArgumentException("Data must be 32-bit aligned");

                var err = srtp_unprotect(this, ptr, ref length);
                if (err == err_status_t.err_status_replay_fail)
                {
                    // Clear out the replay data and try again
                    ClearRtpReplayData(ptr);
                    err = srtp_unprotect(this, ptr, ref length);
                }
                if (err == err_status_t.err_status_auth_fail)
                {
                    // Temporarily increment the ROC and try again (this appears to be a bug in chromium with retransmitted packets near rollovers)
                    err = UnprotectNextRoc(ptr, ref length);
                }
                CheckRtp(err, ptr);
            }
            finally
            {
                h.Free();
            }

            // Make sure length remains valid
            if (length < 0 || (uint)(offset + length) > (uint)data.Length)
                throw new InvalidOperationException("Possible memory corruption in srtp_unprotect");
        }

        private unsafe err_status_t UnprotectNextRoc(IntPtr ptr, ref int length)
        {
            //CheckAuth(err, ptr, data, offset, length);

            // Get the stream context
            if (ptr == IntPtr.Zero)
                return err_status_t.err_status_bad_param;
            var hdr = (srtp_hdr_t*)ptr;

            var stream = GetStream(hdr->ssrc);
            if (stream == null)
                return err_status_t.err_status_no_ctx;

            // Save the current ROC + seq
            var oldIndex = stream->rtp_rdbx.index;
            try
            {
                // Increment the ROC temporarily to see if that fixes the auth issue
                stream->rtp_rdbx.index += 0x10000;
                return srtp_unprotect(this, ptr, ref length);
            }
            finally
            {
                // Restore the real ROC + seq
                stream->rtp_rdbx.index = oldIndex;
            }
        }

        // This needs to change if the crypto policy changes
        public const int MaxChecksumSize = 32;

        public void Protect(byte[] data, int offset, ref int length)
        {
            if ((uint)offset > (uint)data.Length)
                throw new ArgumentOutOfRangeException("offset");
            if (length < 0 || (uint)(offset + length) > (uint)data.Length)
                throw new ArgumentOutOfRangeException("length");

            var limit = (uint)(offset + length + MaxChecksumSize);
            if (limit > (uint)data.Length)
                throw new ArgumentException("Not enough room for checksum");

            var h = GCHandle.Alloc(data, GCHandleType.Pinned);
            try
            {
                var ptr = h.AddrOfPinnedObject() + offset;
                if (((ulong)ptr & 3) != 0)
                    throw new ArgumentException("Data must be 32-bit aligned");

                // Replay checking is disabled for transmit, so no need to clear the replay DB
                CheckRtp(srtp_protect(this, ptr, ref length), ptr);
            }
            finally
            {
                h.Free();
            }

            // Make sure length remains valid
            if (length < 0 || (uint)(offset + length) > (uint)data.Length)
                throw new InvalidOperationException("Possible memory corruption in srtp_protect");

            // Double-check that we reserved enough space
            if ((uint)(offset + length) > limit)
                throw new Exception("MaxChecksumSize too small");
        }

        public static int GetHeaderLength(ByteBuffer buf)
        {
            var extension = (buf[0] & 0x10) != 0;
            var ccidCount = buf[0] & 0xF;

            var len = 12 + 4 * ccidCount;
            if (extension)
            {
                var exLen = buf.ToUInt16BE(len + 2);
                len += 4 + exLen * 4;
            }
            return len;
        }

        public void UnprotectRtcp(byte[] data, int offset, ref int length)
        {
            if ((uint)offset > (uint)data.Length)
                throw new ArgumentOutOfRangeException("offset");
            if (length < 0 || (uint)(offset + length) > (uint)data.Length)
                throw new ArgumentOutOfRangeException("length");

            var h = GCHandle.Alloc(data, GCHandleType.Pinned);
            try
            {
                var ptr = h.AddrOfPinnedObject() + offset;
                if (((ulong)ptr & 3) != 0)
                    throw new ArgumentException("Data must be 32-bit aligned");

                var err = srtp_unprotect_rtcp(this, ptr, ref length);
                if (err == err_status_t.err_status_replay_fail || err == err_status_t.err_status_replay_old)
                {
                    // Clear out the replay data, reset the replay window and try again
                    ClearRtcpReplayData(ptr, length);
                    err = srtp_unprotect_rtcp(this, ptr, ref length);
                }
                CheckRtcp(err, ptr, length);
            }
            finally
            {
                h.Free();
            }

            // Make sure length remains valid
            if (length < 0 || (uint)(offset + length) > (uint)data.Length)
                throw new InvalidOperationException("Possible memory corruption in srtp_unprotect_rtcp");
        }

        public void ProtectRtcp(byte[] data, int offset, ref int length)
        {
            if ((uint)offset > (uint)data.Length)
                throw new ArgumentOutOfRangeException("offset");
            if (length < 0 || (uint)(offset + length) > (uint)data.Length)
                throw new ArgumentOutOfRangeException("length");

            var limit = (uint)(offset + length + MaxChecksumSize);
            if (limit > (uint)data.Length)
                throw new ArgumentException("Not enough room for checksum");

            var h = GCHandle.Alloc(data, GCHandleType.Pinned);
            try
            {
                var ptr = h.AddrOfPinnedObject() + offset;
                if (((ulong)ptr & 3) != 0)
                    throw new ArgumentException("Data must be 32-bit aligned");

                CheckRtcp(srtp_protect_rtcp(this, ptr, ref length), ptr, length);
            }
            finally
            {
                h.Free();
            }

            // Make sure length remains valid
            if (length < 0 || (uint)(offset + length) > (uint)data.Length)
                throw new InvalidOperationException("Possible memory corruption in srtp_protect_rtcp");

            // Double-check that we reserved enough space
            if ((uint)(offset + length) > limit)
                throw new Exception("MaxChecksumSize too small");
        }

        #endregion Public API

        #region Error Handling

        private static void Check(err_status_t err)
        {
            if (err != err_status_t.err_status_ok)
                throw new SrtpException(err);
        }

        private void CheckRtp(err_status_t err, IntPtr packet)
        {
            if (err != err_status_t.err_status_ok)
                throw new SrtpException(err, GetRtpInfo(packet));
        }

        private unsafe string GetRtpInfo(IntPtr packet)
        {
            try
            {
                // Parse out the RTP header
                var hdr = (srtp_hdr_t*)packet;
                if (hdr == null)
                    return "No packet";

                // Find the associated stream
                var stream = GetStream(hdr->ssrc);
                var index = stream != null ? stream->rtp_rdbx.index : 0;

                // Dump the packet header and current stream info
                return string.Format("hdr: {0:X2}{1:X2} seq: {2:X4} ts: {3:X8} ssrc: {4:X8} index: {5:X12}",
                    hdr->b0, hdr->b1, (ushort)IPAddress.NetworkToHostOrder((short)hdr->seq),
                    (uint)IPAddress.NetworkToHostOrder((int)hdr->ts),
                    (uint)IPAddress.NetworkToHostOrder((int)hdr->ssrc),
                    index);
            }
            catch (Exception ex)
            {
                return ex.ToString();
            }
        }

        private void CheckRtcp(err_status_t err, IntPtr packet, int length)
        {
            if (err != err_status_t.err_status_ok)
                throw new SrtpException(err, GetRtcpInfo(packet, length));
        }

        private unsafe string GetRtcpInfo(IntPtr packet, int length)
        {
            try
            {
                // Parse out the RTP header
                var hdr = (srtcp_hdr_t*)packet;
                if (hdr == null || length < 22) // 8-byte header + 4-byte trailer + 10-byte auth tag
                    return "No packet: " + length;

                // Get the sequence number
                var seq = *(uint*)((byte*)packet + length - 14); // 4-byte trailer + 10-byte auth tag
                seq = (uint)IPAddress.NetworkToHostOrder((int)seq) & 0x7FFFFFFF; // MSB is a flag indicating if the packet is encrypted

                // Find the associated stream
                var stream = GetStream(hdr->ssrc);
                var index = stream != null ? stream->rtcp_rdb.window_start : 0;

                // Dump the packet header and current stream info
                return string.Format("hdr: {0:X2}{1:X2} len: {2} ssrc: {3:X8} seq: {4:X16} index: {5:X12}",
                    hdr->b0, hdr->b1, (ushort)IPAddress.NetworkToHostOrder((short)hdr->len),
                    (uint)IPAddress.NetworkToHostOrder((int)hdr->ssrc),
                    seq, index);
            }
            catch (Exception ex)
            {
                return ex.ToString();
            }
        }

        private unsafe void ClearRtpReplayData(IntPtr packet)
        {
            try
            {
                // Parse out the RTP header
                var hdr = (srtp_hdr_t*)packet;
                if (hdr == null)
                    return;

                // Lookup the matching stream
                var stream = GetStream(hdr->ssrc);
                if (stream == null)
                    return;

                // Sanity check
                if (stream->rtp_rdbx.bitmask.word == null || stream->rtp_rdbx.bitmask.length != WindowSize)
                {
                    Logger.Warn("Unexpected rtp_rdbx.bitmask", new
                    {
                        SSRC = (uint)IPAddress.NetworkToHostOrder((int)stream->ssrc),
                        Ptr = (IntPtr)stream->rtp_rdbx.bitmask.word,
                        Length = stream->rtp_rdbx.bitmask.length,
                    });
                    return;
                }

                // Clear RTP sequence db
                for (var i = 0; i < WindowSizeWords; ++i)
                {
                    stream->rtp_rdbx.bitmask.word[i] = 0;
                }
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, "Failed to clear replay data");
            }
        }

        private unsafe void ClearRtcpReplayData(IntPtr packet, int length)
        {
            try
            {
                // Parse out the RTP header
                var hdr = (srtcp_hdr_t*)packet;
                if (hdr == null || length < 22) // 8-byte header + 4-byte trailer + 10-byte auth tag
                    return;

                // Get the sequence number
                var seq = *(uint*)((byte*)packet + length - 14); // 4-byte trailer + 10-byte auth tag
                seq = (uint)IPAddress.NetworkToHostOrder((int)seq) & 0x7FFFFFFF; // MSB is a flag indicating if the packet is encrypted

                // Lookup the matching stream
                var stream = GetStream(hdr->ssrc);
                if (stream == null)
                    return;

                // Clear RTCP sequence db
                stream->rtcp_rdb.bitmask[0] = 0;
                stream->rtcp_rdb.bitmask[1] = 0;

                // Update the replay window to ensure it includes this packet
                if (seq < stream->rtcp_rdb.window_start)
                {
                    stream->rtcp_rdb.window_start = seq >= 128 ? seq - 127 : 0;
                }
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, "Failed to clear RTCP replay data");
            }
        }

        private unsafe srtp_stream_ctx_t* GetStream(uint netssrc) // SSRC in network byte order
        {
            try
            {
                // Make sure this wasn't disposed
                var ctx = (srtp_ctx_t*)handle;
                if (ctx == null)
                    return null;

                // Search the linked-list of streams for a matching SSRC
                var stream = ctx->stream_list;
                while (stream != null)
                {
                    if (stream->ssrc == netssrc)
                    {
                        return stream;
                    }

                    // Check the next stream
                    stream = stream->next;
                }
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, "Error while searching for SRTP stream");
            }
            return null;
        }

        private byte[] _key;
        private HMACSHA1 _auth;

        private unsafe void CheckAuth(err_status_t err, IntPtr ptr, byte[] data, int offset, int length)
        {
            // Find the stream context for this packet
            var hdr = (srtp_hdr_t*)ptr;
            if (hdr == null)
                return;

            var stream = GetStream(hdr->ssrc);
            if (stream == null)
                return;

            // Get the current ROC for this stream
            var roc = (uint)(stream->rtp_rdbx.index >> 16);

            // Extract the auth key
            var keyLen = stream->rtp_auth->key_len;
            if (_key == null || _key.Length != keyLen)
                _key = new byte[keyLen];

            var opad = (byte*)stream->rtp_auth->state; // First field of state is a 64-byte opad
            Marshal.Copy((IntPtr)opad, _key, 0, keyLen);

            for (var i = 0; i < _key.Length; ++i)
                _key[i] ^= 0x5c;

            if (_auth == null)
                _auth = new HMACSHA1();
            _auth.Key = _key;

            // Save the current tag value
            var tag = data.ToUInt32LE(offset + length - 4);
            try
            {
                // Auth check with different ROC values to see if any of them validate
                data.SetBytesBE(roc - 1, offset + length - 4);
                var minus1 = _auth.ComputeHash(data, offset, length);

                data.SetBytesBE(roc, offset + length - 4);
                var equal = _auth.ComputeHash(data, offset, length);

                data.SetBytesBE(roc + 1, offset + length - 4);
                var plus1 = _auth.ComputeHash(data, offset, length);

                // Log the various auth tags for comparison
                Logger.Debug("Auth Check " + err, new
                {
                    Tag = tag.ToString("X8"),
                    Minus1 = BitConverter.ToUInt32(minus1, 0).ToString("X8"),
                    Equal = BitConverter.ToUInt32(equal, 0).ToString("X8"),
                    Plus1 = BitConverter.ToUInt32(plus1, 0).ToString("X8"),
                    Info = GetRtpInfo(ptr),
                });
            }
            finally
            {
                // Reset the tag bytes
                data.SetBytesLE(tag, offset + length - 4);
            }
        }

        #endregion Error Handling

        #region Native

        private const int SRTP_MASTER_KEY_LENGTH = 16;
        private const int SRTP_MASTER_SALT_LENGTH = 14;
        private const int SRTP_MASTER_LENGTH = (SRTP_MASTER_KEY_LENGTH + SRTP_MASTER_SALT_LENGTH);

        // Stuff from the srtp.h header

        //private const int SRTP_MASTER_KEY_LEN = 30;
        //private const int SRTP_MAX_KEY_LEN = 64;
        //private const int SRTP_MAX_TAG_LEN = 16;
        //private const int SRTP_MAX_TRAILER_LEN = SRTP_MAX_TAG_LEN;

        //private const int SRTP_AEAD_SALT_LEN = 12;
        //private const int AES_128_GCM_KEYSIZE_WSALT = SRTP_AEAD_SALT_LEN + 16;
        //private const int AES_192_GCM_KEYSIZE_WSALT = SRTP_AEAD_SALT_LEN + 24;
        //private const int AES_256_GCM_KEYSIZE_WSALT = SRTP_AEAD_SALT_LEN + 32;

        private enum sec_serv_t
        {
            sec_serv_none = 0,
            sec_serv_conf = 1,
            sec_serv_auth = 2,
            sec_serv_conf_and_auth = 3,
        }

        [StructLayout(LayoutKind.Sequential)]
        private struct crypto_policy_t
        {
            public uint cipher_type;
            public int cipher_key_len;
            public uint auth_type;
            public int auth_key_len;
            public int auth_tag_len;
            public sec_serv_t sec_serv;
        }

        private enum ssrc_type_t
        {
            ssrc_undefined = 0,
            ssrc_specific = 1,
            ssrc_any_inbound = 2,
            ssrc_any_outbound = 3,
        }

        [StructLayout(LayoutKind.Sequential)]
        private struct ssrc_t
        {
            public ssrc_type_t type;
            public uint value;
        }

        [StructLayout(LayoutKind.Sequential)]
        private struct srtp_policy_t
        {
            public ssrc_t ssrc;
            public crypto_policy_t rtp;
            public crypto_policy_t rtcp;
            public IntPtr key;
            public IntPtr ekt;
            public uint window_size;
            public int allow_repeat_tx;
            public IntPtr next;
        }

        public enum err_status_t
        {
            err_status_ok = 0,  // nothing to report
            err_status_fail = 1,  // unspecified failure
            err_status_bad_param = 2,  // unsupported parameter
            err_status_alloc_fail = 3,  // couldn't allocate memory
            err_status_dealloc_fail = 4,  // couldn't deallocate properly
            err_status_init_fail = 5,  // couldn't initialize
            err_status_terminus = 6,  // can't process as much data as requested
            err_status_auth_fail = 7,  // authentication failure
            err_status_cipher_fail = 8,  // cipher failure
            err_status_replay_fail = 9,  // replay check failed (bad index)
            err_status_replay_old = 10, // replay check failed (index too old)
            err_status_algo_fail = 11, // algorithm failed test routine
            err_status_no_such_op = 12, // unsupported operation
            err_status_no_ctx = 13, // no appropriate context found
            err_status_cant_check = 14, // unable to perform desired validation
            err_status_key_expired = 15, // can't use key any more
            err_status_socket_err = 16, // error in use of socket
            err_status_signal_err = 17, // error in use POSIX signals
            err_status_nonce_bad = 18, // nonce check failed
            err_status_read_fail = 19, // couldn't read data
            err_status_write_fail = 20, // couldn't write data
            err_status_parse_err = 21, // error parsing data
            err_status_encode_err = 22, // error encoding data
            err_status_semaphore_err = 23,// error while using semaphores
            err_status_pfkey_err = 24  // error while using pfkey
        }

        [DllImport(SRTPDLL, CallingConvention = CallingConvention.Cdecl)]
        private static extern err_status_t srtp_init();

        [DllImport(SRTPDLL, CallingConvention = CallingConvention.Cdecl)]
        private static extern void crypto_policy_set_aes_cm_128_hmac_sha1_32(ref crypto_policy_t p);

        [DllImport(SRTPDLL, CallingConvention = CallingConvention.Cdecl, EntryPoint = "crypto_policy_set_rtp_default")]
        private static extern void crypto_policy_set_aes_cm_128_hmac_sha1_80(ref crypto_policy_t p);

        [DllImport(SRTPDLL, CallingConvention = CallingConvention.Cdecl)]
        private static extern err_status_t srtp_create(out Srtp session, ref srtp_policy_t policy);

        [DllImport(SRTPDLL, CallingConvention = CallingConvention.Cdecl)]
        [SuppressUnmanagedCodeSecurity]
        private static extern err_status_t srtp_dealloc(IntPtr session);

        [DllImport(SRTPDLL, CallingConvention = CallingConvention.Cdecl)]
        private static extern err_status_t srtp_protect(Srtp session, IntPtr rtp_hdr, ref int len_ptr);

        [DllImport(SRTPDLL, CallingConvention = CallingConvention.Cdecl)]
        private static extern err_status_t srtp_unprotect(Srtp session, IntPtr srtp_hdr, ref int len_ptr);

        [DllImport(SRTPDLL, CallingConvention = CallingConvention.Cdecl)]
        private static extern err_status_t srtp_remove_stream(Srtp session, uint ssrc);

        [DllImport(SRTPDLL, CallingConvention = CallingConvention.Cdecl)]
        private static extern err_status_t srtp_protect_rtcp(Srtp session, IntPtr rtcp_hdr, ref int len_ptr);

        [DllImport(SRTPDLL, CallingConvention = CallingConvention.Cdecl)]
        private static extern err_status_t srtp_unprotect_rtcp(Srtp session, IntPtr srtcp_hdr, ref int len_ptr);

        [StructLayout(LayoutKind.Sequential)]
        private unsafe struct srtp_ctx_t
        {
            public srtp_stream_ctx_t *stream_list;     /* linked list of streams            */
            public srtp_stream_ctx_t *stream_template; /* act as template for other streams */
            public void *user_data;                    /* user custom data */
        }

        private const int SRTP_AEAD_SALT_LEN = 12;

        [StructLayout(LayoutKind.Sequential)]
        private unsafe struct srtp_stream_ctx_t
        {
            public uint            ssrc;
            public srtp_cipher_t  *rtp_cipher;
            //public srtp_cipher_t  *rtp_xtn_hdr_cipher;
            public srtp_auth_t    *rtp_auth;
            public srtp_rdbx_t     rtp_rdbx;
            public srtp_sec_serv_t rtp_services;
            public srtp_cipher_t  *rtcp_cipher;
            public srtp_auth_t    *rtcp_auth;
            public srtp_rdb_t      rtcp_rdb;
            public srtp_sec_serv_t rtcp_services;
            public srtp_key_limit_ctx_t *limit;
            public direction_t     direction;
            public int             allow_repeat_tx;
            public srtp_ekt_stream_ctx_t *ekt; 
            public fixed byte      salt[SRTP_AEAD_SALT_LEN];   /* used with GCM mode for SRTP */
            public fixed byte      c_salt[SRTP_AEAD_SALT_LEN]; /* used with GCM mode for SRTCP */
            //public int            *enc_xtn_hdr;
            //public int             enc_xtn_hdr_count;
            public srtp_stream_ctx_t *next;   /* linked list of streams */
        }

        [StructLayout(LayoutKind.Sequential)]
        private struct srtp_rdbx_t
        {            
            public ulong index;
            public bitvector_t bitmask;
        }

        [StructLayout(LayoutKind.Sequential)]
        private unsafe struct bitvector_t
        {
            public uint length;   
            public uint *word;
        }

        [StructLayout(LayoutKind.Sequential)]
        private unsafe struct srtp_rdb_t
        {
            public uint window_start; /* packet index of the first bit in bitmask */
            public fixed ulong bitmask[2];
        }

        private enum srtp_sec_serv_t
        {
            sec_serv_none = 0, /**< no services                        */
            sec_serv_conf = 1, /**< confidentiality                    */
            sec_serv_auth = 2, /**< authentication                     */
            sec_serv_conf_and_auth = 3  /**< confidentiality and authentication */
        }

        private enum direction_t
        {
            dir_unknown       = 0,
            dir_srtp_sender   = 1, 
            dir_srtp_receiver = 2
        }

        private struct srtp_ekt_stream_ctx_t
        {
            // TODO
        }

        private struct srtp_cipher_t
        {
            // TODO
        }

        [StructLayout(LayoutKind.Sequential)]
        private unsafe struct srtp_auth_t
        {
            public srtp_auth_type_t *type;
            public void        *state;
            public int out_len;                  /* length of output tag in octets */
            public int key_len;                  /* length of key in octets        */
            public int prefix_len;               /* length of keystream prefix     */
        }

        private struct srtp_auth_type_t
        {
            // TODO
        }

        private struct srtp_key_limit_ctx_t
        {
            // TODO
        }

        [StructLayout(LayoutKind.Sequential)]
        private struct srtp_hdr_t
        {
            public byte b0; // |V=2|P|X|  CC  |
            public byte b1; // |M|     PT     |
            public ushort seq;
            public uint ts;
            public uint ssrc;
        }

        [StructLayout(LayoutKind.Sequential)]
        private struct srtcp_hdr_t
        {
            public byte b0; // |V=2|P|    RC   |
            public byte b1; // |   PT=SR=200   |
            public ushort len;
            public uint ssrc;
        }

        #endregion Native
    }
}
