'''
Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.

Licensed under the Apache License, Version 2.0 (the "License").
You may not use this file except in compliance with the License.
A copy of the License is located at

    http://aws.amazon.com/apache2.0/

or in the "license" file accompanying this file.
This file is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
either express or implied. See the License for
the specific language governing permissions
and limitations under the License.
'''

# delete_old_db_dest_aurora
# This lambda function will delete any database that matches PATTERN except the most recent one.
# Set PATTERN to a regex that matches your Aurora cluster identifiers (by default: aurora-snapshot-tool)
# Set DEST_REGION to the destination AWS region
import boto3
import os
import logging
import snapshots_tool_utils as util

# Initialize everything
DEST_REGION = os.getenv('DEST_REGION', os.getenv('AWS_DEFAULT_REGION')).strip()
LOGLEVEL = os.getenv('LOG_LEVEL', 'INFO').strip()
PATTERN = 'ss-tool-'

# This next line is for debug only, comment it out:
# logging.basicConfig()

logger = logging.getLogger()
logger.setLevel(LOGLEVEL.upper())


def lambda_handler(event, context):

    client = boto3.client('rds', region_name=DEST_REGION)
    clusters = util.paginate_api_call(client, 'describe_db_clusters', 'DBClusters')
    clusters = util.filter_clusters(PATTERN, clusters)
    count = len(clusters)

    if count < 2:
        logger.info('No databases to delete. Count: {0}'.format(count))
        return

    i = 0
    for db_cluster in sorted(clusters, key=lambda x: x['ClusterCreateTime']):
        i = i + 1

        if i == count:
            logger.info('Not deleting final database: {0} ({1}) created: {2}'.format(
                    db_cluster['DBClusterIdentifier'], db_cluster['Status'],
                    db_cluster['ClusterCreateTime']))
            # Do not delete the newest database.
            return

        if db_cluster['Status'] != "deleting":
            logger.info('Deleting database: {0} ({1}) created: {2}'.format(
                db_cluster['DBClusterIdentifier'], db_cluster['Status'],
                db_cluster['ClusterCreateTime']))
            client.delete_db_cluster(
                DBClusterIdentifier=db_cluster['DBClusterIdentifier'],
                SkipFinalSnapshot=True,
            )
            # check response? try/catch? hm..
            continue

        logger.info("Old Database still being deleted: {0}".format(
            db_cluster['DBClusterIdentifier']))


if __name__ == '__main__':
    lambda_handler(None, None)
