package main

import (
	"context"
	"encoding/json"
	"fmt"
	"log"
	"net/http"
	"strconv"
	"time"

	"github.com/nexucis/grafana-go-client/api/types"
)

const (
	defaultText    = "Jenkins Annotation"
	annotationsAPI = "/api/annotations"
)

var (
	errDashboardID = fmt.Errorf("invalid or no dashboard ID provided")
)

// Annotation contains the POST JSON and update ID (if it's an update).
type Annotation struct {
	Grafana *Grafana
	Data    json.RawMessage
	PostAnnotations
}

type PostAnnotations struct {
	AnnotationID int64    `json:"annotationID"`
	DashboardID  int64    `json:"dashboardId"`
	PanelID      int64    `json:"panelId"`
	Time         int64    `json:"time"`
	Text         string   `json:"text,omitempty"`
	Tags         []string `json:"tags,omitempty"`
	IsRegion     bool     `json:"isRegion"`
	TimeEnd      int64    `json:"timeEnd"`
}

// Run is the main handler for annotations.
func (a *Annotation) Run(ctx context.Context) (Output, error) {
	err := json.Unmarshal(a.Data, &a.PostAnnotations)
	if err != nil {
		return clientError(err, http.StatusBadRequest)
	}

	msg := "Updated Annotation"

	switch a.AnnotationID {
	case 0:
		msg = "Created Annotation"
		a.AnnotationID, err = a.Create(ctx)
	default:
		err = a.Update(ctx)
	}

	if err != nil {
		return serverError(err)
	}

	msg = fmt.Sprintf("%s ID %d for dashboard ID %d, tags: %v, msg: %s", msg, a.AnnotationID,
		a.PostAnnotations.DashboardID, a.PostAnnotations.Tags, a.PostAnnotations.Text)
	log.Print(msg)

	// Return a response with a 200 OK status and the ID.
	return Output{Msg: msg, ID: a.AnnotationID}, nil
}

// Create creates a brand new annotation and returns the ID.
func (a *Annotation) Create(ctx context.Context) (int64, error) {
	if a.DashboardID == 0 {
		return 0, errDashboardID
	}

	if a.Time == 0 {
		a.Time = time.Now().UnixNano() / int64(time.Millisecond)
	}

	if a.Text == "" {
		a.Text = defaultText
	}

	obj := types.ResponseCreateAnnotation{}
	req := a.Grafana.Post(annotationsAPI).Body(a.PostAnnotations)
	err := req.Context(ctx).Do().SaveAsObj(&obj)

	return obj.ID, err
}

// Update updates and existing annotation by ID.
// Automatically updates end time to now if not provided.
func (a *Annotation) Update(ctx context.Context) error {
	if a.TimeEnd == 0 {
		a.TimeEnd = time.Now().UnixNano() / int64(time.Millisecond)
	}

	return a.Grafana.Patch(annotationsAPI).
		SetSubPath("/:id").
		SetPathParam("id", strconv.FormatInt(a.AnnotationID, 10)).
		Body(a.PostAnnotations).Context(ctx).Do().Error()
}
