package main

import (
	"context"
	"encoding/json"
	"fmt"
	"log"
	"net/http"
	"strconv"

	"github.com/nexucis/grafana-go-client/api/types"
)

const (
	dashboardsAPI = "/api/dashboards/db"
	dashboardUID  = "/api/dashboards/uid/"
	dashboardFind = "/api/search"
)

// Dashboard represents the input data.
type Dashboard struct {
	Grafana *Grafana
	Data    json.RawMessage
	PostDashboard
}

// PostDashboard is the data wen send to Grafana's API, converted from the raw input data.
type PostDashboard struct {
	Dashboard DashboardData `json:"dashboard"`
	Refresh   string        `json:"refresh"`
	Message   string        `json:"message"`
	FolderID  int64         `json:"folderId"`
	Overwrite bool          `json:"overwrite"`
}

// DashboardReply is the data Grafana returns when updating a dashboard.
type DashboardReply struct {
	ID      int64               `json:"id"`
	Slug    string              `json:"slug"`
	Status  string              `json:"status"`
	UID     string              `json:"uid"`
	URL     string              `json:"url"`
	Version int                 `json:"version"`
	Meta    types.DashboardMeta `json:"meta"`
}

// DashboardData represents the dashboard payload. This will change over time.
type DashboardData struct {
	ID            int64       `json:"id"`
	UID           string      `json:"uid"`
	Title         string      `json:"title"`
	Version       int64       `json:"version"`
	Tags          []string    `json:"tags,omitempty"`
	Timezone      string      `json:"timezone,omitempty"`
	SchemaVersion int64       `json:"schemaVersion"`
	Editable      bool        `json:"editable,omitempty"`
	GraphToolTip  int         `json:"graphToolTip,omitempty"`
	Iteration     int64       `json:"iteration,omitempty"`
	Description   string      `json:"description,omitempty"`
	Refresh       string      `json:"refresh,omitempty"`
	Style         string      `json:"style,omitempty"`
	GnetID        int64       `json:"gnetId,omitempty"`
	Panels        interface{} `json:"panels,omitempty"`
	Time          interface{} `json:"time,omitempty"`
	TimePicker    interface{} `json:"timepicker,omitempty"`
	TimeOptions   interface{} `json:"time_options,omitempty"`
	Templating    interface{} `json:"templating,omitempty"`
	Rows          interface{} `json:"rows,omitempty"`
	Annotations   interface{} `json:"annotations,omitempty"`
	Links         interface{} `json:"links,omitempty"`
}

// Update is the main handler for dashboards, it updates an existing dashboard.
func (d *Dashboard) Update(ctx context.Context) (Output, error) {
	var resp DashboardReply

	if err := json.Unmarshal(d.Data, &d.PostDashboard); err != nil {
		return clientError(err, http.StatusBadRequest)
	} else if d.Dashboard.ID < 1 {
		return clientError(errDashboardID, http.StatusPreconditionFailed)
	} else if err := d.SetFolderID(ctx); err != nil {
		return serverError(err)
	}

	req := d.Grafana.Post(dashboardsAPI).Body(d.PostDashboard)
	if err := req.Context(ctx).Do().SaveAsObj(&resp); err != nil {
		return serverError(err)
	}

	msg := fmt.Sprintf("Updated Dashboard %s://%s%s ID %d, UID %s, Version %d, Folder %d, Status: %s",
		d.Grafana.BaseURL.Scheme, d.Grafana.BaseURL.Host, resp.URL,
		resp.ID, resp.UID, resp.Version, d.FolderID, resp.Status)
	log.Print(msg)

	return Output{
		ID:      resp.ID,
		Version: resp.Version,
		UID:     resp.UID,
		Msg:     msg,
	}, nil
}

// SetFolderID sets the folder ID for a dashboard POST request.
// If the folder ID already exists, then this makes sure the dashboard exists.
func (d *Dashboard) SetFolderID(ctx context.Context) error {
	var (
		resp []types.DashboardMeta
		req  = d.Grafana.Get(dashboardFind).
			AddQueryParam("dashboardIds", strconv.FormatInt(d.Dashboard.ID, 10)).
			Context(ctx).Do()
		err = req.SaveAsObj(&resp)
	)

	if len(resp) > 0 && d.FolderID < 1 && d.Dashboard.ID > 0 {
		d.FolderID = resp[0].FolderID
	}

	return err
}

// GetID converts a dashboard UID into a dashboard ID, and returns the version.
func (d *Dashboard) GetID(ctx context.Context) (Output, error) {
	var (
		from struct{ UID string }
		resp struct {
			Meta      types.DashboardMeta
			Dashboard DashboardData
		}
	)

	if err := json.Unmarshal(d.Data, &from); err != nil {
		return clientError(err, http.StatusBadRequest)
	}

	req := d.Grafana.Get(dashboardUID + from.UID)
	if err := req.Context(ctx).Do().SaveAsObj(&resp); err != nil {
		return serverError(err)
	}

	return Output{
		ID:      resp.Dashboard.ID,
		Version: resp.Meta.Version,
		UID:     resp.Dashboard.UID,
		Msg:     resp.Dashboard.Title,
	}, nil
}
