package main

import (
	"context"
	"fmt"
	"net"
	"net/http"
	"net/url"
	"os"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/endpoints"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/secretsmanager"
	"github.com/nexucis/grafana-go-client/grafanahttp"
)

var (
	errNoGrafanaURL = fmt.Errorf("no grafana URL provided (env:%s)", grafanaAPIURLEnvVar)
	errNoGrafanaKey = fmt.Errorf("no grafana API Key provided (env:%s)", grafanaAPIKeyEnvVar)
)

// Grafana contains the grafana HTTP client and methods to do things with it.
type Grafana struct {
	*grafanahttp.RESTClient
}

// NewGrafana creates the HTTP client for Grafana.
func NewGrafana(grafanaURL, grafanaKey string) (*Grafana, error) {
	if grafanaURL == "" {
		return nil, errNoGrafanaURL
	} else if grafanaKey == "" {
		return nil, errNoGrafanaKey
	}

	u, err := url.Parse(grafanaURL)
	if err != nil {
		err = fmt.Errorf("%w: invalid Grafana URL", err)
	}

	return &Grafana{RESTClient: &grafanahttp.RESTClient{
		Token:   grafanaKey,
		BaseURL: u,
		Client: &http.Client{Transport: &http.Transport{
			DialContext: (&net.Dialer{
				Timeout:   defaultTimeout,
				KeepAlive: 0,
			}).DialContext,
			TLSHandshakeTimeout: defaultTimeout}},
	}}, err
}

// GetAPIToken fetches the Grafana API token from AWS secrets manager.
func GetAPIToken(ctx context.Context, name string) (string, error) {
	region := os.Getenv("AWS_REGION")
	if region == "" {
		region = defaultAWSRegion
	}

	svc := secretsmanager.New(session.Must(session.NewSession(&aws.Config{
		STSRegionalEndpoint: endpoints.RegionalSTSEndpoint,
		Region:              aws.String(region),
	})))

	result, err := svc.GetSecretValueWithContext(ctx, &secretsmanager.GetSecretValueInput{SecretId: aws.String(name)})
	if err != nil || result == nil || result.SecretString == nil {
		return "", err
	}

	return *result.SecretString, nil
}
