package main

import (
	"context"
	"encoding/json"
	"fmt"
	"log"
	"net/http"
	"os"
	"strings"
	"time"

	"github.com/aws/aws-lambda-go/lambda"
	"github.com/aws/aws-sdk-go/aws/endpoints"
)

// Env Var names for URL and API Key.
const (
	grafanaAPIKeySMEVar = "GRAFANA_API_KEY_SM_NAME"
	grafanaAPIKeyEnvVar = "GRAFANA_API_KEY"
	grafanaAPIURLEnvVar = "GRAFANA_API_URL"
	defaultTimeout      = 5 * time.Second
	defaultAWSRegion    = endpoints.UsWest2RegionID // used if AWS_REGION is not set.
)

var (
	errUnknownCmd = fmt.Errorf("unknown or invalid command provided")
	errLog        = log.New(os.Stderr, "ERROR ", log.Llongfile)
)

// Lambda is the data needed to update Grafana via API.
type Lambda struct {
	grafanaURL string // Grafana URL to update
	grafanaKey string // Grafana API Key (plaintext), or ...
	secretName string // AWS Secrets Manager secret name for: Grafana [Editor] API Key
}

// Input is the lambda input.
type Input struct {
	Cmd  string          `json:"cmd"`
	Data json.RawMessage `json:"data"`
}

// Output is the data this app prints out when it's done.
type Output struct {
	Msg     string `json:"message"`
	ID      int64  `json:"id"`
	UID     string `json:"uid,omitempty"`
	Version int    `json:"version,omitempty"`
}

func main() {
	l := &Lambda{
		grafanaKey: os.Getenv(grafanaAPIKeyEnvVar),
		secretName: os.Getenv(grafanaAPIKeySMEVar),
		grafanaURL: os.Getenv(grafanaAPIURLEnvVar),
	}

	if len(os.Args) > 1 && os.Args[1] == "-local" {
		l.RunLocal()
		return
	}

	lambda.Start(l.Handler)
}

// RunLocal is for CLI use.
func (l *Lambda) RunLocal() {
	var payload Input

	if err := json.NewDecoder(os.Stdin).Decode(&payload); err != nil {
		log.Fatal(err)
	}

	output, err := l.Handler(context.Background(), payload)
	if err != nil {
		log.Fatalln(err)
	}

	log.Printf("ID: %v, UID: %v, Version: %v, Msg: %v",
		output.ID, output.UID, output.Version, output.Msg)
}

// Handler is the main lambda Handler.
// Takes care of annotation creation and updates or other things.
func (l *Lambda) Handler(ctx context.Context, payload Input) (Output, error) {
	ctx, cancel := context.WithTimeout(ctx, defaultTimeout)
	defer cancel()

	if l.grafanaKey == "" {
		var err error

		l.grafanaKey, err = GetAPIToken(ctx, l.secretName)
		if err != nil {
			return serverError(err)
		}
	}

	grafana, err := NewGrafana(l.grafanaURL, l.grafanaKey)
	if err != nil {
		return serverError(err)
	}

	switch cmd := strings.ToLower(payload.Cmd); cmd {
	// Create or update an annotation.
	case "annotate", "annotation":
		a := &Annotation{Data: payload.Data, Grafana: grafana}
		return a.Run(ctx)
	case "dashboard", "dashboards", "update":
		d := &Dashboard{Data: payload.Data, Grafana: grafana}
		return d.Update(ctx)
	case "getdash", "getdashboard", "getid":
		d := &Dashboard{Data: payload.Data, Grafana: grafana}
		return d.GetID(ctx)
	default:
		return serverError(fmt.Errorf("%w: %s", errUnknownCmd, cmd))
	}
}

// serverError is a helper function to reply with a server error.
func serverError(err error) (Output, error) {
	msg := http.StatusText(http.StatusInternalServerError) + ": " + err.Error()

	errLog.Println(msg)

	return Output{Msg: msg}, nil
}

// clientError is a helper function to reply with a client error.
func clientError(err error, status int) (Output, error) {
	msg := http.StatusText(status)

	if err != nil {
		errLog.Println("client error:", err)
		msg += ": " + err.Error()
	}

	return Output{Msg: msg}, nil
}
