package datasource

import (
	"context"
	"fmt"
	"log"
	"net"
	"net/http"
	"net/url"
	"os"
	"strings"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/client"
	"github.com/aws/aws-sdk-go/aws/request"
	"github.com/aws/aws-sdk-go/service/secretsmanager"
	"github.com/aws/aws-sdk-go/service/sns"
	"github.com/nexucis/grafana-go-client/api"
	"github.com/nexucis/grafana-go-client/grafanahttp"
)

const timeout = 10 * time.Second

// App errors.
var (
	ErrNoGrafanaURL  = fmt.Errorf("no grafana URL provided")
	ErrNoGrafanaKey  = fmt.Errorf("no grafana API Key provided")
	ErrWrongOwner    = fmt.Errorf("the existing datasource does not belong to the requesting account")
	ErrNoAccountID   = fmt.Errorf("no account ID found in payload")
	ErrNoAccountName = fmt.Errorf("no account name found in payload")
	ErrUnknownReq    = fmt.Errorf("unknown request type")
)

// Config makes this module go brrrr.
// Careful what goes in here. This may be re-used for _different_ requests.
type Config struct {
	GrafanaURL   string
	GrafanaToken string
	realToken    string
	SNSToken     string
	SNS          SNS
	SM           SecretsManager
	Grafana      api.ClientInterface
}

// New returns a new lambda handler.
func New(sess client.ConfigProvider, awsc *aws.Config) *Config {
	c := &Config{
		GrafanaURL:   strings.TrimSpace(os.Getenv("GRAFANA_URL")),
		GrafanaToken: strings.TrimSpace(os.Getenv("GRAFANA_TOKEN")), // secret key
		SNSToken:     strings.TrimSpace(os.Getenv("SNS_TOKEN")),     // secret key
		SM:           secretsmanager.New(sess, awsc),                // get secrets
		SNS:          sns.New(sess, awsc),                           // send SNS (slack)
	}

	ctx, cancel := context.WithTimeout(context.Background(), timeout)
	defer cancel()

	err := c.setToken(ctx)
	if err != nil {
		log.Println("[ERROR] Getting Grafana Token from Secrets Manager:", err)
	}

	c.Grafana, err = newGrafana(c.GrafanaURL, c.realToken)
	if err != nil {
		log.Println("[ERROR] Getting Grafana Client:", err)
	}

	return c
}

// newGrafana creates the HTTP client for Grafana.
func newGrafana(grafanaURL, key string) (api.ClientInterface, error) {
	if grafanaURL == "" {
		return nil, ErrNoGrafanaURL
	} else if key == "" {
		return nil, ErrNoGrafanaKey
	}

	u, err := url.Parse(grafanaURL)
	if err != nil {
		return nil, fmt.Errorf("%w: invalid Grafana URL", err)
	}

	d := &net.Dialer{Timeout: timeout, KeepAlive: 0}

	return api.NewWithClient(&grafanahttp.RESTClient{Token: key, BaseURL: u, Client: &http.Client{
		Timeout:   timeout,
		Transport: &http.Transport{DialContext: d.DialContext, TLSHandshakeTimeout: timeout},
	}}), nil
}

// SecretsManager allows mocking Secrets lookups for tests.
type SecretsManager interface {
	GetSecretValueWithContext(context.Context, *secretsmanager.GetSecretValueInput,
		...request.Option) (*secretsmanager.GetSecretValueOutput, error)
}

func (c *Config) setToken(ctx context.Context) error {
	if c.realToken != "" {
		return nil
	}

	secret, err := c.SM.GetSecretValueWithContext(ctx, &secretsmanager.GetSecretValueInput{
		VersionId:    nil,
		VersionStage: nil,
		SecretId:     aws.String(c.GrafanaToken),
	})
	if err != nil || secret.SecretString == nil {
		return fmt.Errorf("getting API key/token '%s' from secrets manager: %w", c.GrafanaToken, err)
	}

	c.realToken = *secret.SecretString

	return nil
}
