package datasource

import (
	"context"
	"fmt"
	"strconv"
	"strings"

	"github.com/nexucis/grafana-go-client/api/types"
)

const (
	base10 = 10
	bits64 = 64
	zero   = "0"
)

// datasourceCheck gets a datasource from Grafana and validates the name matches the requesting account.
// This is the crux of our security; disallowing someone from deleting (or updating) another account's datasource.
//nolint:unparam
func (c *Config) datasourceCheck(ctx context.Context, req *Request, id int64) (*types.DataSource, error) {
	datasource, err := c.Grafana.DataSources().GetByID(id)
	if err != nil {
		return nil, fmt.Errorf("checking data source %d: %w", id, err)
	}

	if !strings.Contains(datasource.Name, req.AccountID) {
		return nil, fmt.Errorf("%w: name '%s' for id %d does not contain '%s'",
			ErrWrongOwner, datasource.Name, id, req.AccountID)
	}

	return datasource, nil
}

//nolint:stylecheck,exhaustivestruct
func (c *Config) datasourceCreate(ctx context.Context, req *Request) error {
	existingID, _ := c.Grafana.DataSources().GetIDByName(req.DatasourceName())
	if existingID > 0 {
		req.DSourceID = strconv.FormatInt(existingID, base10)
		req.Note = fmt.Sprintf("Datasource `%s` already exists: _%s_, imported and updated",
			req.DatasourceName(), req.DSourceID)

		return c.datasourceUpdate(ctx, req)
	}

	add := &types.AddDataSource{
		IsDefault: false,
		Access:    Proxy,
		URL:       req.DatasourceURL(true),
		Type:      req.DatasourceType(),
		Name:      req.DatasourceName(),
		JSONData:  req.CloudWatchDS(),
	}

	sm, err := c.Grafana.DataSources().Create(add)
	if err != nil {
		return fmt.Errorf("Creating data source: %w", err)
	}

	req.ID = strconv.FormatInt(sm.ID, base10)

	return nil
}

//nolint:stylecheck,exhaustivestruct
func (c *Config) datasourceUpdate(ctx context.Context, req *Request) error {
	id, err := strconv.ParseInt(req.DSourceID, base10, bits64)
	if err != nil {
		return fmt.Errorf("Dashboard ID (`physicalResourceID`) is not a number: %w", err)
	}

	ds, err := c.datasourceCheck(ctx, req, id)
	if err != nil {
		return fmt.Errorf("Ownership validation failed: %w", err)
	}

	req.url = ds.URL
	update := types.UpdateDataSource{
		IsDefault: false,
		Access:    Proxy,
		URL:       req.DatasourceURL(true),
		Type:      req.DatasourceType(),
		Name:      req.DatasourceName(),
		JSONData:  req.CloudWatchDS(),
	}

	sm, err := c.Grafana.DataSources().Update(id, update)
	if err != nil {
		return fmt.Errorf("Updating data source: %w", err)
	}

	req.ID = strconv.FormatInt(sm.ID, base10)

	return nil
}

//nolint:stylecheck
func (c *Config) datasourceDelete(ctx context.Context, req *Request) error {
	if req.DSourceID == zero || req.DSourceID == "" {
		return nil
	}

	id, err := strconv.ParseInt(req.DSourceID, base10, bits64)
	if err != nil {
		return fmt.Errorf("Dashboard ID (`physicalResourceID`) is not a number: %w", err)
	}

	if _, err := c.datasourceCheck(ctx, req, id); err != nil {
		if strings.Contains(strings.ToLower(err.Error()), "data source not found") {
			req.Note = fmt.Sprintf("Datasource `%s` was already deleted/missing: _%d_", req.DatasourceName(), id)
			return nil //nolint:nlreturn
		}

		return fmt.Errorf("Ownership validation failed: %w", err)
	}

	if err := c.Grafana.DataSources().Delete(id); err != nil {
		return fmt.Errorf("Deleting data source failed: %w", err)
	}

	req.ID = req.DSourceID

	return nil
}
