package datasource

import (
	"context"
	"fmt"
	"log"
	"strings"

	"github.com/aws/aws-lambda-go/cfn"
	"github.com/aws/aws-sdk-go/aws"
)

const (
	// regionLoc is used to find the region in the Stack ID.
	regionLoc = 3
	// accountLoc is used to find the Account ID in the Stack ID.
	accountLoc = 4
)

// LambdaHandler checks the payload and creates the datasource in Grafana.
// https://godoc.org/github.com/aws/aws-lambda-go/cfn
// == You must always return a physical resource ID, even if there is an error.
// == Otherwise, the error does not make it back to the caller.
func (c *Config) LambdaHandler(ctx aws.Context, data cfn.Event) (string, map[string]interface{}, error) {
	values := c.PrintLog(&data)
	if values["name"] == "" {
		return zero, nil, ErrNoAccountName
	}

	err := c.setToken(ctx)
	if err != nil {
		return zero, nil, err
	}

	if c.Grafana == nil {
		if c.Grafana, err = newGrafana(c.GrafanaURL, c.realToken); err != nil {
			return zero, nil, err
		}
	}

	stackSplit := strings.Split(data.StackID, ":")
	if !(len(stackSplit) > accountLoc) {
		// This is rather unlikely, and can only happen on a forged request.
		return zero, nil, ErrNoAccountID
	}

	return c.finishRequest(ctx, &Request{
		AccountName: values["name"],          // Freetype. Requested Isengard name.
		Region:      stackSplit[regionLoc],   // 3: Used as default region for datasource.
		AccountID:   stackSplit[accountLoc],  // 4: Isengard ID.
		Type:        data.RequestType,        // Create, update, delete.
		DSourceID:   data.PhysicalResourceID, // Only set on update or delete.
		Metrics:     values["metrics"],       // Empty is allowed.
		Role:        values["role"],          // Empty is allowed, has a default.
		OldName:     values["oldname"],       // Only set on update or delete.
		AddTime:     strings.EqualFold(values["time"], "true"),
		AddXRay:     strings.EqualFold(values["xray"], "true"),
		ID:          zero,
		Error:       "",
	})
}

// PrintLog writes a log entry about this incoming request.
// It also returns the payload resource properties.
func (c *Config) PrintLog(data *cfn.Event) map[string]string {
	v := make(map[string]string)
	v["name"], _ = data.ResourceProperties["AccountName"].(string)
	v["metrics"], _ = data.ResourceProperties["Metrics"].(string)
	v["role"], _ = data.ResourceProperties["IAMRole"].(string)
	v["xray"], _ = data.ResourceProperties["AddXRay"].(string)
	v["time"], _ = data.ResourceProperties["AddTime"].(string)
	v["oldname"], _ = data.OldResourceProperties["AccountName"].(string)

	switch data.RequestType {
	case cfn.RequestCreate:
		log.Printf("Create Request: Account Name: %s, Metrics: %s, Role: %s, X-Ray: %s, Timestream: %s, StackID: %s\n",
			v["name"], v["metrics"], v["role"], v["xray"], v["time"], data.StackID)
	case cfn.RequestDelete:
		log.Printf("Delete Request: Account Name: %s, Metrics: %s, Role: %s, X-Ray: %s, "+
			"Timestream: %s, StackID: %s, Datasource ID: %s\n",
			v["name"], v["metrics"], v["role"], v["xray"], v["time"], data.StackID, data.PhysicalResourceID)
	case cfn.RequestUpdate:
		log.Printf("Update Request: Account Name: %s, StackID: %s, Datasource ID: %s, "+
			"new Account Name: %s, Metrics: %s, Role: %s, X-Ray: %s, Timestream: %s\n",
			v["oldname"], data.StackID, data.PhysicalResourceID, v["name"], v["metrics"], v["role"], v["xray"], v["time"])
	default:
		log.Printf("Unknown Request: %s, StackID: %s, ResourceProperties: %v\n",
			data.ResourceType, data.StackID, data.ResourceProperties)
	}

	return v
}

func (c *Config) finishRequest(ctx context.Context, req *Request) (string, map[string]interface{}, error) {
	defer c.PublishSNS(ctx, req)

	var err error
	// The functions below may change req.ID.
	switch req.Type {
	default:
		err = fmt.Errorf("%w: %s", ErrUnknownReq, req.Type)
	case cfn.RequestCreate:
		err = c.datasourceCreate(ctx, req)
	case cfn.RequestUpdate:
		err = c.datasourceUpdate(ctx, req)
	case cfn.RequestDelete:
		err = c.datasourceDelete(ctx, req)
	}

	if req.Note != "" {
		log.Printf("NOTICE: %s", req.Note)
	}

	if err != nil {
		req.Error = err.Error() // this goes to SNS->Slack.
		return req.ID, nil, err // nolint:nlreturn
	}

	log.Printf("%sd data source with ID %v. Datasource Name/Type: %v/%v",
		req.Type, req.ID, req.AccountName, req.DatasourceType())

	return req.ID, req.Data(), nil
}
