package datasource

import (
	"fmt"
	"strings"
	"time"

	"github.com/aws/aws-lambda-go/cfn"
)

// Pieces of a datasource.
const (
	IsengardURL = "https://isengard.amazon.com/account/"
	Cloudwatch  = "cloudwatch"
	Timestream  = "grafana-timestream-datasource"
	XRay        = "grafana-x-ray-datasource"
	Proxy       = "proxy"
	// DefaultIAMRole is IAM role name used if none is provided.
	DefaultIAMRole = "grafana-cloudwatch-read-only"
)

// Request represents a processable incoming request.
type Request struct {
	AccountName string          `json:"accountName"`
	AccountID   string          `json:"accountId"`
	Region      string          `json:"region"`
	DSourceID   string          `json:"datasourceId"`
	Type        cfn.RequestType `json:"type"`
	Metrics     string          `json:"metrics"`
	Role        string          `json:"role"`
	AddXRay     bool            `json:"xray"`
	AddTime     bool            `json:"time"`
	OldName     string          `json:"oldName"`
	Error       string          `json:"error"`
	Note        string          `json:"note"`
	ID          string          `json:"id"`
	url         string
}

// CloudWatchDS represents the extra data in a cloudwatch datasource.
type CloudWatchDS struct {
	Arn     string `json:"assumeRoleArn"`
	Auth    string `json:"authType"`
	Region  string `json:"defaultRegion"`
	Time    string `json:"timeField"`
	Metrics string `json:"customMetricsNamespaces,omitempty"`
}

// CloudWatchDS returns the json "extra" data for the CloudWatch datasource.
func (r *Request) CloudWatchDS() *CloudWatchDS {
	if r.Role == "" {
		r.Role = DefaultIAMRole
	}

	role := r.Role
	if !strings.HasPrefix(role, "arn:aws:iam") {
		role = fmt.Sprintf("arn:aws:iam::%s:role/%s", r.AccountID, role)
	}

	metrics := strings.ReplaceAll(r.Metrics, " ", "")
	if metrics == "-" || r.AddTime || r.AddXRay {
		// Metrics are only for cloudwatch datasources.
		metrics = ""
	}

	return &CloudWatchDS{
		Arn:     role,
		Auth:    "default",
		Region:  r.Region,
		Time:    "@timestamp",
		Metrics: metrics,
	}
}

// Data returns the request data in a format that SNS can use.
func (r *Request) Data() map[string]interface{} {
	return map[string]interface{}{
		"ID":   r.ID,
		"Name": r.DatasourceName(),
		"Type": r.DatasourceType(),
	}
}

// DatasourceURL creates a meta tag for the data source. This is only for "display" purposes.
func (r *Request) DatasourceURL(includeDate bool) string {
	const (
		added   = "added"
		updated = "updated"
		before  = "before-2021-08"
		format  = "%s - %s %s - %s %s"
	)

	realURL := IsengardURL + r.AccountID
	if !includeDate { // return a slack-usable URL.
		return realURL
	}

	today := time.Now().Format("2006-01-02")

	if r.url == "" { // It's brand new.
		return fmt.Sprintf(format, realURL, added, today, updated, today)
	}

	// It already exists, so (attempt to) extract the "added" date from the existing URL.
	fields := strings.Fields(r.url)
	if len(fields) < 4 || fields[2] != added {
		// no added date, make a fresh URL.
		return fmt.Sprintf(format, realURL, added, before, updated, today)
	}

	return fmt.Sprintf(format, realURL, added, fields[3], updated, today)
}

// DatasourceType returns the datasource type for Grafana.
// Grafana uses some weird names, so set them from constants.
func (r *Request) DatasourceType() string {
	switch {
	default:
		return Cloudwatch
	case r.AddTime:
		return Timestream
	case r.AddXRay:
		return XRay
	}
}

// DatasourceName returns the name of the datasource.
// Removes spaces and @amazon.com suffix.
// Three different supported datasource types, set the name accordingly.
// Spaces are trimmed from beginning and end, then replaced with underscores throughout.
func (r *Request) DatasourceName() string {
	name := strings.ReplaceAll(strings.TrimSuffix(strings.TrimSpace(r.AccountName), "@amazon.com"), " ", "_")

	switch {
	default:
		return fmt.Sprintf("%s %s", name, r.AccountID)
	case r.AddTime:
		return fmt.Sprintf("%s Timestream %s", name, r.AccountID)
	case r.AddXRay:
		return fmt.Sprintf("%s XRay %s", name, r.AccountID)
	}
}
