package slackhook

import (
	"context"
	"fmt"
	"strings"

	"git.xarth.tv/Xarth-Grafana/cloudwatch-datasource/pkg/datasource"
	"github.com/aws/aws-lambda-go/cfn"
)

const (
	// This value goes in the footer.
	sourceURL = "https://git.xarth.tv/Xarth-Grafana/cloudwatch-datasource"
	wikiURL   = "https://wiki.xarth.tv/x/3oDaBw"
	// Side-bar colors.
	red   = "#b30c12" // red
	green = "#00cc00" // green
	teal  = "#00cc8b" // blue
	pink  = "#b30c8c" // purplish
	// Icons (make space at the end).
	pfxError   = ":ducksiren: "
	pfxError2  = ":deadglitch: "
	pfxDelete  = ":fire_upsidedown: "
	pfxCreate  = ":homer: "
	pfxUpdate  = ":lab_coat: "
	pfxUnknown = ":interrobang: "
)

// NotifyDatasourceChanges process notification from a custom resource.
func (c *Config) NotifyDatasourceChanges(ctx context.Context, req *datasource.Request) error {
	var (
		prefix       = pfxUnknown
		color        = red
		header       = &Text{Text: fmt.Sprintf("%sd Datasource", req.Type)}
		fields, text = c.getFields(req)
		footer       = fmt.Sprintf(":infraplat: From %s via %s, %s", Link(c.GrafanURL, c.BotName),
			Link(sourceURL, "CloudWatch Datasource Automation"), Link(wikiURL, "Wiki"))
	)

	switch {
	case req.Error != "":
		fields = append(fields, &Field{Title: "ERROR", Value: pfxError2 + "*" + req.Error + "*"})
		prefix = pfxError
		header.Text = "Error " + strings.TrimSuffix(string(req.Type), "e") + "ing Datasource!"
		text.Text += "*Error*: " + pfxError2 + req.Error + "\n"
	case req.Type == cfn.RequestDelete:
		prefix, color = pfxDelete, pink
	case req.Type == cfn.RequestCreate:
		prefix, color = pfxCreate, green
	case req.Type == cfn.RequestUpdate:
		prefix, color = pfxUpdate, teal
	default:
		// This probably can't happen.
		header.Text = "Unknown Request: " + string(req.Type)
	}

	return c.Send(ctx, "", &BlkHeader{Text: header},
		&BlkSection{Text: &Text{Text: prefix + "*Datasource Name*: `" + req.DatasourceName() + "`"}},
		&Attachment{
			Fallback: text.Text,
			Color:    color,
			Fields:   fields,
			Footer:   footer,
		},
	)
}

// Fields are the meta data. The Text returned is shown on mobile devices or IRC, etc.
func (c *Config) getFields(req *datasource.Request) ([]*Field, *Text) {
	var (
		text   = &Text{Text: fmt.Sprintf("*Role*: %s\n*Region*: %s\n*Datasource ID*: `%s`\n", req.Role, req.Region, req.ID)}
		fields = []*Field{
			{Title: "Account Name", Value: req.AccountName, Short: true},
			{Title: "IAM Role", Value: req.Role, Short: true},
			{Title: "AWS Region", Value: req.Region, Short: true},
			{Title: "Grafana Plugin", Value: req.DatasourceType(), Short: true},
			{Title: "Isengard ID", Value: Link(req.DatasourceURL(false), req.AccountID), Short: true},
		}
	)

	if req.ID != "0" && req.ID != "" {
		fields = append(fields, &Field{
			Title: "Datasource ID",
			Value: Link(c.GrafanURL+"/datasources/edit/"+req.ID, req.ID),
			Short: true,
		})
	}

	if req.OldName != "" && req.OldName != req.AccountName {
		fields = append(fields, &Field{Title: "Old Name", Value: req.OldName, Short: true})
		text.Text += "*Old Name*:\n" + req.OldName
	}

	if req.Metrics != "" && req.Metrics != "-" {
		fields = append(fields, &Field{
			Short: false,
			Title: "Custom Metric Namespaces",
			Value: "• " + strings.ReplaceAll(req.Metrics, ",", "\n • "),
		})
		text.Text += "*Metrics*:\n" + req.Metrics + "\n"
	}

	if req.Note != "" {
		fields = append(fields, &Field{
			Short: false,
			Title: "Note",
			Value: req.Note,
		})
		text.Text += "*Note*:\n" + req.Note + "\n"
	}

	return fields, text
}
