// Package slackhook sends Slack messages from an SNS Topic.
//nolint:tagliatelle
package slackhook

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"strings"
)

// Custom errors this library may return.
var (
	ErrInvalidBlockType = fmt.Errorf("unknown block type provided")
	ErrInvalidResp      = fmt.Errorf("invalid HTTP response code")
)

//nolint:gochecknoglobals
var httpClient = &http.Client{Timeout: timeout}

// BlockType locks the types to specific values.
type BlockType string

// ElemType locks the types to specific values.
type ElemType string

// Element represents an element.
type Element interface{}

// Various BlockTypes that can be created.
const (
	BTActions BlockType = "actions"
	BTContext BlockType = "context"
	BTDivider BlockType = "divider"
	BTFile    BlockType = "file"
	BTHeader  BlockType = "header"
	BTImage   BlockType = "image"
	BTInput   BlockType = "input"
	BTSection BlockType = "section"
)

// Varous ElemTypes this library natively supports.
const (
	ETButton     ElemType = "button"
	ETTimePicker ElemType = "timepicker"
)

// Message is the payload we use in a Webhook POST.
type Message struct {
	Username    string        `json:"username,omitempty"`
	IconEmoji   string        `json:"icon_emoji,omitempty"`
	Channel     string        `json:"channel,omitempty"`
	Text        string        `json:"text,omitempty"`
	Blocks      []interface{} `json:"blocks,omitempty"`
	Attachments []*Attachment `json:"attachments,omitempty"`
}

// BlkActions is an action block.
type BlkActions struct {
	Type     BlockType `json:"type"`
	BlockID  string    `json:"block_id,omitempty"`
	Elements []Element `json:"elements,omitempty"`
}

// BlkContext is an context block.
type BlkContext struct {
	Type     BlockType `json:"type"`
	BlockID  string    `json:"block_id,omitempty"`
	Elements []Element `json:"elements,omitempty"`
}

// BlkDivider is a divider block.
type BlkDivider struct {
	Type    BlockType `json:"type"`
	BlockID string    `json:"block_id,omitempty"`
}

// BlkFile is a file block.
type BlkFile struct {
	Type       BlockType `json:"type"`
	BlockID    string    `json:"block_id,omitempty"`
	ExternalID string    `json:"external_id,omitempty"`
	Source     string    `json:"source,omitempty"`
}

// BlkHeader is a header block.
type BlkHeader struct {
	Type    BlockType `json:"type"`
	BlockID string    `json:"block_id,omitempty"`
	Text    *Text     `json:"text,omitempty"`
}

// BlkImage is an image block.
type BlkImage struct {
	Type     BlockType `json:"type"`
	BlockID  string    `json:"block_id,omitempty"`
	Text     string    `json:"alt_text,omitempty"`
	Title    *Text     `json:"title,omitempty"`
	ImageURL string    `json:"image_url,omitempty"`
}

// BlkInput is an input block.
type BlkInput struct {
	Type           BlockType   `json:"type"`
	BlockID        string      `json:"block_id,omitempty"`
	Label          *Text       `json:"label,omitempty"`
	Element        interface{} `json:"element,omitempty"`
	Hint           *Text       `json:"hint,omitempty"`
	DispatchAction bool        `json:"dispatch_action,omitempty"`
	Optional       bool        `json:"optional,omitempty"`
}

// BlkSection is a section block.
type BlkSection struct {
	Type      BlockType   `json:"type"`
	BlockID   string      `json:"block_id,omitempty"`
	Text      *Text       `json:"text,omitempty"`
	Accessory interface{} `json:"accessory,omitempty"`
	Fields    []*Text     `json:"fields,omitempty"`
}

// TextType represents the possible Text Types.
type TextType string

// Only two text types, and image is shoehorned in.
const (
	TextPlain    TextType = "plain_text"
	TextMarkdown TextType = "mrkdwn"
	ImageImage   TextType = "image"
)

// Image is part of specific Blocks, like Section.
type Image struct {
	Type     TextType `json:"type"`
	Text     string   `json:"alt_text,omitempty"`
	ImageURL string   `json:"image_url,omitempty"`
}

// Text is part of specific Blocks, like Section.
type Text struct {
	Type     TextType `json:"type"`
	Text     string   `json:"text,omitempty"`
	Emoji    bool     `json:"emoji,omitempty"`
	Verbatim bool     `json:"verbatim,omitempty"`
}

// Attachment is incomplete; there's more pieces and it's legacy, so wasn't used..
type Attachment struct {
	Fallback   string        `json:"fallback,omitempty"`
	Color      string        `json:"color,omitempty"`
	Fields     []*Field      `json:"fields,omitempty"`
	Blocks     []interface{} `json:"blocks,omitempty"`
	Footer     string        `json:"footer,omitempty"`
	FooterIcon string        `json:"footer_icon,omitempty"`
}

// ElemButton represents a button element.
type ElemButton struct {
	Type     ElemType `json:"type"`            // "button"
	Text     Text     `json:"text"`            // "Click Me"
	Value    string   `json:"value"`           // "click_me"
	Style    string   `json:"style,omitempty"` // "primary" (green), "danger" (red), (blank) (neutral)
	URL      string   `json:"url,omitempty"`   // optional: "http://..."
	ActionID string   `json:"action_id,omitempty"`
}

// ElemTimePicker represents a timepicker element.
type ElemTimePicker struct {
	Type        ElemType `json:"type"`         // "timepicker"
	Placeholder Text     `json:"placeholder"`  // "Pick Time"
	InitialTime string   `json:"initial_time"` // "13:37"
	ActionID    string   `json:"action_id,omitempty"`
}

// Field is part of an attachment.
type Field struct {
	Title string `json:"title,omitempty"`
	Value string `json:"value,omitempty"`
	Short bool   `json:"short,omitempty"`
}

// Send POSTs a Slack Message to a Slack Webhook URL.
func Send(ctx context.Context, url string, msg *Message) ([]byte, error) {
	body, err := json.Marshal(msg)
	if err != nil {
		return body, fmt.Errorf("marshaling: %w", err)
	}

	req, err := http.NewRequestWithContext(ctx, http.MethodPost, url, bytes.NewBuffer(body))
	if err != nil {
		return body, fmt.Errorf("creating req: %w", err)
	}

	resp, err := httpClient.Do(req)
	if err != nil {
		return body, fmt.Errorf("making req: %w", err)
	}
	defer resp.Body.Close()

	b, _ := ioutil.ReadAll(resp.Body)

	if resp.StatusCode < 200 || resp.StatusCode > 299 {
		return body, fmt.Errorf("POSTing to Slack: %w: %s, resp: %s", ErrInvalidResp, resp.Status, string(b))
	}

	return body, nil
}

// CheckBlocks makes sure the blocks are OK and converts them into attachments and blocks.
func CheckBlocks(blocks []interface{}) ([]*Attachment, []interface{}, error) {
	pa, pb := make([]*Attachment, 0), make([]interface{}, 0)

	for _, b := range blocks {
		switch block := b.(type) {
		case *Attachment:
			pa = append(pa, block)

			if _, _, err := CheckBlocks(block.Blocks); err != nil {
				return nil, nil, err
			}
		default:
			pb = append(pb, block)

			if err := checkBlock(block); err != nil {
				return nil, nil, err
			}
		}
	}

	return pa, pb, nil
}

func checkElements(elements []Element) {
	for _, element := range elements {
		switch element := element.(type) {
		case *ElemTimePicker:
			element.Type = ETTimePicker
			if element.Placeholder.Type == "" {
				element.Placeholder.Type = TextPlain
			}
		case *ElemButton:
			element.Type = ETButton
			if element.Value == "" {
				element.Value = strings.ReplaceAll(strings.ToLower(element.Text.Text), " ", "_")
			}

			if element.Text.Type == "" {
				element.Text.Type = TextPlain
			}
		}
	}
}

func checkBlock(block interface{}) error { //nolint:cyclop
	switch block := block.(type) {
	default:
		return fmt.Errorf("%w: %T", ErrInvalidBlockType, block)
	case *BlkActions:
		checkElements(block.Elements)
		block.Type = BTActions
	case *BlkContext:
		checkElements(block.Elements)
		block.Type = BTContext

		for _, t := range block.Elements {
			switch m := t.(type) {
			case *Text:
				m.Type = TextMarkdown
			case *Image:
				m.Type = ImageImage
			}
		}
	case *BlkDivider:
		block.Type = BTDivider
	case *BlkFile:
		block.Type = BTFile
	case *BlkHeader:
		block.Type = BTHeader
		if block.Text != nil {
			block.Text.Type = TextPlain
		}
	case *BlkImage:
		block.Type = BTImage
	case *BlkInput:
		block.Type = BTInput
	case *BlkSection:
		block.Type = BTSection
		if block.Text != nil && block.Text.Type == "" {
			block.Text.Type = TextMarkdown
		}

		for _, t := range block.Fields {
			if t != nil && t.Type == "" {
				t.Type = TextMarkdown
			}
		}
	}

	return nil
}

// Link makes a slack text link.
func Link(url, title string) string {
	return "<" + url + "|" + title + ">"
}
