package slackhook

import (
	"context"
	"fmt"
	"log"
	"os"
	"strings"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/client"
	"github.com/aws/aws-sdk-go/aws/request"
	"github.com/aws/aws-sdk-go/service/secretsmanager"
)

const timeout = 10 * time.Second

// Config for the Slack package.
type Config struct {
	realURL   string
	URL       string
	Channel   string
	BotName   string
	Emoji     string
	GrafanURL string
	SM        SecretsManager
}

// New returns a slack config to send messages. Designed for Lambda.
func New(sess client.ConfigProvider, awsc *aws.Config) *Config {
	c := &Config{
		GrafanURL: strings.TrimSpace(os.Getenv("GRAFANA_URL")),
		URL:       strings.TrimSpace(os.Getenv("SLACK_HOOK")),
		Channel:   strings.TrimSpace(os.Getenv("SLACK_CHAN")),
		BotName:   strings.TrimSpace(os.Getenv("SLACK_NAME")),
		Emoji:     strings.TrimSpace(os.Getenv("SLACK_EMOJI")),
		SM:        secretsmanager.New(sess, awsc),
	}

	ctx, cancel := context.WithTimeout(context.Background(), timeout)
	defer cancel()

	if err := c.setURL(ctx); err != nil {
		log.Println("[ERROR] Getting Slack URL from Secrets Manager:", err)
	}

	return c
}

// Send POSTs a webhook message to Slack.
func (c *Config) Send(ctx context.Context, msg string, blocks ...interface{}) error {
	attachments, blocks, err := CheckBlocks(blocks)
	if err != nil {
		return err
	}

	body, err := Send(ctx, c.realURL, &Message{
		Username:    c.BotName,
		IconEmoji:   c.Emoji,
		Channel:     c.Channel,
		Text:        msg,
		Blocks:      blocks,
		Attachments: attachments,
	})

	log.Printf("Sent %d byte Slack message to %s: %s", len(body), c.Channel, string(body))

	if err != nil {
		return err
	}

	return nil
}

// SecretsManager allows mocking Secrets lookups for tests.
type SecretsManager interface {
	GetSecretValueWithContext(context.Context, *secretsmanager.GetSecretValueInput,
		...request.Option) (*secretsmanager.GetSecretValueOutput, error)
}

func (c *Config) setURL(ctx context.Context) error {
	if c.realURL != "" {
		return nil
	}

	secret, err := c.SM.GetSecretValueWithContext(ctx, &secretsmanager.GetSecretValueInput{
		VersionStage: nil,
		VersionId:    nil,
		SecretId:     aws.String(c.URL),
	})
	if err != nil || secret.SecretString == nil {
		return fmt.Errorf("getting URL '%s' from secrets manager: %w", c.URL, err)
	}

	c.realURL = *secret.SecretString

	return nil
}
