package main

import (
	"encoding/json"
	"io/ioutil"
	"log"
	"net/http"
	"os"
	"path/filepath"
	"strings"

	"github.com/tidwall/pretty"
)

// WriteDashboards downloads and saves all the dashboard from Grafana.
func (f *Flags) WriteDashboards() error {
	dashes, err := f.getDashboards()
	if err != nil {
		return err
	}

	log.Printf("Processing %d dashboards.", len(dashes))

	return f.writeFiles(dashes)
}

func (f *Flags) getDashboards() ([]Dashboard, error) {
	log.Printf("Pulling %s/api/search\n", f.URL)

	res, err := http.Get(f.URL + "/api/search")
	if err != nil {
		return nil, err
	}
	defer res.Body.Close()

	var v []Dashboard

	return v, json.NewDecoder(res.Body).Decode(&v)
}

func (f *Flags) writeFiles(dashes []Dashboard) error {
	if _, err := createFolder(f.Folder); err != nil {
		return err
	}

	for _, dash := range dashes {
		if dash.Type != DashboardType {
			continue
		}

		folder, err := createFolder(f.Folder, sanitizeName(dash.FolderTitle))
		if err != nil {
			return err
		}

		path := filepath.Join(folder, dash.UID+".json")
		if err := f.writeDashboardFile(dash.UID, path); err != nil {
			return err
		}
	}

	return nil
}

func (f *Flags) writeDashboardFile(uid, path string) error {
	log.Printf("Pulling %s/api/dashboards/uid/%s\n", f.URL, uid)

	res, err := http.Get(f.URL + "/api/dashboards/uid/" + uid)
	if err != nil {
		return err
	}
	defer res.Body.Close()

	log.Println("Writing File:", path)

	b, err := ioutil.ReadAll(res.Body)
	if err != nil {
		return err
	}

	w, err := os.Create(path)
	if err != nil {
		return err
	}

	_, err = w.Write(pretty.Pretty(b))

	return err
}

func createFolder(parts ...string) (string, error) {
	name := filepath.Join(parts...)

	_, err := os.Stat(name)
	if os.IsNotExist(err) {
		log.Println("Creating Folder:", name)

		err = os.MkdirAll(name, 0755)
	}

	return name, err
}

// sanitizeName replaces bad characters with an underscore.
// Used before creating a folder with this name.
func sanitizeName(name string) string {
	name = strings.TrimSpace(name)

	bad := []string{" ", "/", `\`, "!", "@", "#", "$", "%", "^", "&", "*", "(", ")", "~", "`", `"`, "[", "]", "{", "}"}
	for _, c := range bad {
		name = strings.ReplaceAll(name, c, "_")
	}

	return name
}
