package main

import (
	"flag"
	"fmt"
	"log"
	"os"
	"time"
)

const (
	// Folder is the place to write files.
	Folder = "dashes"
	// DefaultURL is the instance URL to work with.
	DefaultURL = "https://grafana.xarth.tv"
	// DashboardType specifies a dashboard versus a folder which has a different type.
	DashboardType = "dash-db"
)

var (
	errNoDashboardUIDs = fmt.Errorf("must provide 1 or more dashboard uids as arguments")
)

// Dashboard represents a normal dashboard payload from the api/search method.
type Dashboard struct {
	ID          int      `json:"id"`
	UID         string   `json:"uid"`
	Title       string   `json:"title"`
	URI         string   `json:"uri"`
	URL         string   `json:"url"`
	Slug        string   `json:"slub"`
	Type        string   `json:"type"`
	Tags        []string `json:"tags"`
	FolderID    int      `json:"folderId"`
	FolderUID   string   `json:"folderUid"`
	FolderTitle string   `json:"folderTitle"`
	FolderURL   string   `json:"folderUrl"`
	Version     int      `json:"version"`
	UpdatedBy   string   `json:"updatedBy"`
	CreatedBy   string   `json:"createdBy"`
}

// Flags contains cli args.
type Flags struct {
	Folder string
	URL    string
	Cmd    string
	Delim  string
	Prefix string
	Suffix string
	*flag.FlagSet
}

// ParseFlags turns cli args into structured data.
func ParseFlags(args []string) *Flags {
	f := &Flags{FlagSet: flag.NewFlagSet("flags", flag.ExitOnError)}

	f.StringVar(&f.Folder, "folder", Folder, "Folder path where dashboard backups are written.")
	f.StringVar(&f.URL, "url", DefaultURL, "Grafana URL.")
	f.StringVar(&f.Cmd, "cmd", "dashboards", "'dashboards' does backups and 'meta' prints dashboard data.")
	f.StringVar(&f.Delim, "delim", ",", "meta delimeter.")
	f.StringVar(&f.Prefix, "prefix", "", "meta prefix.")
	f.StringVar(&f.Suffix, "suffix", "", "meta suffix.")

	_ = f.Parse(args)

	return f
}

func main() {
	start := time.Now()
	flags := ParseFlags(os.Args[1:])

	if err := flags.Run(); err != nil {
		log.Fatalln("ERROR:", err, "Elapsed:", time.Since(start).Round(time.Second))
	}

	log.Println("Done. Elapsed:", time.Since(start).Round(time.Second))
}

// Run does the thing.
func (f *Flags) Run() error {
	switch f.Cmd {
	default:
		f.Usage()
		return fmt.Errorf("Unknown Command: %s", f.Cmd)
	case "meta":
		return f.PrintMeta(f.Args())
	case "dashboards":
		return f.WriteDashboards()
	}
}
