package main

import (
	"fmt"
	"log"
	"strings"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/ssm"
	"github.com/nexucis/grafana-go-client/api/types"
)

// SSMParamSlackKey is the SSM path in the twitch-grafana-prod account where the parameter is stored.
const SSMParamSlackKey = "/twitch-grafana-prod/slack_alert_url.txt"

// Alert allows us to use an interface to write less code.
type Alert interface {
	Render() *types.CreateAlertNotification
}

/* We only support the follow notification channels */

// AlertEmail notification channel for email messages.
type AlertEmail struct {
	UploadImage bool   `json:"uploadImage"`
	Addresses   string `json:"addresses"`
}

// AlertSlack notification channel for slack channels or users.
type AlertSlack struct {
	WebHookURL  string `json:"url"`
	Username    string `json:"username"`
	Recipient   string `json:"recipient"`
	IconEmoji   string `json:"icon_emoji"`
	IconURL     string `json:"icon_url"`
	Mention     string `json:"mention"`
	UploadImage bool   `json:"uploadImage"`
}

// AlertPagerDuty notification channel for pager duty incidents.
type AlertPagerDuty struct {
	name        string
	Key         string `json:"integrationKey"`
	AutoResolve bool   `json:"autoResolve"`
	UploadImage bool   `json:"uploadImage"`
}

// Errors generated by this file.
var (
	ErrDuplicateSlack = fmt.Errorf("notification channel with this name already exists")
	ErrDuplicatePD    = fmt.Errorf("pagerduty integration key already exists")
)

// DoAlerts runs the code to add a new alert channel.
func (g *Graken) DoAlerts() error {
	if g.AlertSlack.Recipient != "" {
		// We have a slack channel! Make sure we can get the Slack API Key.
		var err error
		if g.AlertSlack.WebHookURL, err = g.GetSlackKey(); err != nil {
			return err
		}
	}

	if keys, names, err := g.GetPDKeysAndNames(); err != nil {
		return err
	} else if name, ok := keys[g.AlertPagerDuty.Key]; ok {
		return fmt.Errorf("cannot add '%s': %w '%s': %s",
			g.AlertPagerDuty.name, ErrDuplicatePD, g.AlertPagerDuty.Key, name)
	} else if err = g.CreateAlertNotification(g.AlertSlack, names); err != nil {
		return err
	} else if err = g.CreateAlertNotification(g.AlertPagerDuty, names); err != nil {
		return err
	} else if err = g.CreateAlertNotification(g.AlertEmail, names); err != nil {
		return err
	}

	return nil
}

// CreateAlertNotification calls the grafana api and prints a success message.
func (g *Graken) CreateAlertNotification(alert Alert, names map[string]bool) error {
	cc := alert.Render()
	if cc == nil {
		return nil // no data provided, so no error
	}

	if names[cc.Name] {
		return fmt.Errorf("%w: %s", ErrDuplicateSlack, cc.Name)
	}

	nc, err := g.client.AlertNotifications().Create(cc)
	if err != nil {
		return fmt.Errorf("creating notification channel: %w", err)
	}

	log.Printf("Notification Channel Added! ID: %v, Type: %s, Name: %v", nc.ID, cc.Type, cc.Name)

	return nil
}

// Render turns pagerduty input data into a Grafana AlertNotification.
func (a AlertPagerDuty) Render() *types.CreateAlertNotification {
	if a.name == "" || a.Key == "" {
		return nil // no data provided
	}

	a.AutoResolve = true
	a.UploadImage = true

	return &types.CreateAlertNotification{
		IsDefault: false,
		Name:      a.name,
		Settings:  a,
		Type:      "pagerduty",
	}
}

// Render turns email input data into a Grafana AlertNotification.
func (a AlertEmail) Render() *types.CreateAlertNotification {
	if a.Addresses == "" {
		return nil // no data provided
	}

	a.UploadImage = true

	return &types.CreateAlertNotification{
		IsDefault: false,
		Name:      a.Addresses,
		Settings:  a,
		Type:      "email",
	}
}

// Render turns slack input data into a Grafana AlertNotification.
func (a AlertSlack) Render() *types.CreateAlertNotification {
	if a.Recipient == "" {
		return nil // no data provided
	}

	a.IconEmoji = "grafanaxarth"
	a.Username = "Xarth Grafana"
	a.Recipient = "#" + strings.TrimPrefix(a.Recipient, "#")
	a.UploadImage = true
	a.Mention = ":siren:"

	return &types.CreateAlertNotification{
		IsDefault: false,
		Name:      a.Recipient,
		Settings:  a,
		Type:      "slack",
	}
}

// GetSlackKey grabs the slack api url (key) from an existing (manually edited) data source.
// This allows us to store the key in Grafana and use this app to copy it to a new Slack integration.
func (g *Graken) GetSlackKey() (string, error) {
	param, err := ssm.New(g.awsSess, g.awsClient.WithRegion("us-west-2")).
		GetParameter(&ssm.GetParameterInput{
			Name:           aws.String(SSMParamSlackKey),
			WithDecryption: aws.Bool(false),
		})
	if err != nil {
		return "", fmt.Errorf("make sure you pass an AWS profile for twitch-grafana-prod: %w", err)
	}

	return strings.TrimSpace(param.String()), nil
}

// GetPDKeysAndNames returns a list of all the pagerduty integration keys in Grafana.
// Also returned is a list of all alert notification channel names in Grafana.
// This data is used to make sure we add no duplicates.
func (g *Graken) GetPDKeysAndNames() (map[string]string, map[string]bool, error) {
	nc, err := g.client.AlertNotifications().Get()
	if err != nil {
		return nil, nil, fmt.Errorf("getting existing notifications: %w", err)
	}

	keys := make(map[string]string)
	names := make(map[string]bool)

	for _, n := range nc {
		names[n.Name] = true

		if n.Type != "pagerduty" {
			continue
		}

		settings, ok := n.Settings.(map[string]interface{})
		if !ok || settings["integrationKey"] == nil || settings["integrationKey"].(string) == "" {
			continue
		}

		keys[settings["integrationKey"].(string)] = n.Name
	}

	return keys, names, nil
}
