package main

import (
	"fmt"
	"log"
	"strings"

	"github.com/aws/aws-sdk-go/aws/endpoints"
	"github.com/nexucis/grafana-go-client/api/types"
)

type cloudWatchDS struct {
	Arn     string `json:"assumeRoleArn"`
	Auth    string `json:"authType"`
	Region  string `json:"defaultRegion"`
	Time    string `json:"timeField"`
	Metrics string `json:"customMetricsNamespaces"`
}

// Errors generated by this file.
var (
	ErrDataSourceNotAdded = fmt.Errorf("data source was not added")
	ErrDataSourceExists   = fmt.Errorf("data source already exists")
)

// DoDataSource handles creating a new data source.
func (g *Graken) DoDataSource() error {
	// Name is "twitch-central-aws 007917851548"
	g.Name += " " + g.ID

	if _, err := g.client.DataSources().GetByName(g.Name); err == nil {
		return fmt.Errorf("%w: %s", ErrDataSourceExists, g.Name)
	}

	if g.Email == "" {
		g.Email = "https://isengard.amazon.com/account/" + g.ID
	}

	regions := []string{
		// common regions first.
		endpoints.UsWest2RegionID, // US West (Oregon).
		endpoints.UsEast2RegionID, // US East (Ohio).
		endpoints.UsEast1RegionID, // US East (N. Virginia).
		endpoints.EuWest1RegionID, // Europe (Ireland).
		// uncommon next.
		endpoints.AfSouth1RegionID,     // Africa (Cape Town).
		endpoints.ApEast1RegionID,      // Asia Pacific (Hong Kong).
		endpoints.ApNortheast1RegionID, // Asia Pacific (Tokyo).
		endpoints.ApNortheast2RegionID, // Asia Pacific (Seoul).
		endpoints.ApSouth1RegionID,     // Asia Pacific (Mumbai).
		endpoints.ApSoutheast1RegionID, // Asia Pacific (Singapore).
		endpoints.ApSoutheast2RegionID, // Asia Pacific (Sydney).
		endpoints.CaCentral1RegionID,   // Canada (Central).
		endpoints.EuCentral1RegionID,   // Europe (Frankfurt).
		endpoints.EuNorth1RegionID,     // Europe (Stockholm).
		endpoints.EuSouth1RegionID,     // Europe (Milan).
		endpoints.EuWest2RegionID,      // Europe (London).
		endpoints.EuWest3RegionID,      // Europe (Paris).
		endpoints.MeSouth1RegionID,     // Middle East (Bahrain).
		endpoints.SaEast1RegionID,      // South America (Sao Paulo).
		endpoints.UsWest1RegionID,      // US West (N. California).
	}
	defaultRegion := endpoints.UsWest2RegionID

	// Choose a "more correct" default region for this data source
	// if the data source has a region name _in it_.
	// If no region name exists, choose us-west-2 as default.
	for _, region := range regions {
		if strings.Contains(strings.ToLower(g.Name), region) {
			defaultRegion = region
			break
		}
	}

	datasource := &types.AddDataSource{ //nolint:exhaustivestruct
		Type:   "cloudwatch",
		Name:   g.Name,
		Access: "proxy",
		URL:    g.Email,
		JSONData: cloudWatchDS{
			Arn:     "arn:aws:iam::" + g.ID + ":role/" + g.Role,
			Metrics: strings.ReplaceAll(g.Metrics, " ", ""),
			Auth:    "default",
			Region:  defaultRegion,
			Time:    "@timestamp",
		},
	}

	return g.AddDS(datasource)
}

// AddDS creates the data source in Grafana.
func (g *Graken) AddDS(ds *types.AddDataSource) error {
	sm, err := g.client.DataSources().Create(ds)
	if err != nil {
		return fmt.Errorf("creating data source: %w", err)
	}

	if sm.ID == 0 {
		return fmt.Errorf("%w: %s", ErrDataSourceNotAdded, sm.Message)
	}

	log.Printf("Data Source Added! ID: %v Name: %v", sm.ID, ds.Name)

	return nil
}
