package main

import (
	"flag"
	"fmt"
	"log"
	"net/url"
	"os"

	"git.xarth.tv/Xarth-Grafana/graken/midway"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/nexucis/grafana-go-client/api"
	"github.com/nexucis/grafana-go-client/grafanahttp"
)

const (
	defaultGrafanaURL = "https://grafana.xarth.tv"
	ldapEmailSuffix   = "ldap.twitch.a2z.com"
)

// Graken is our main struct to pass data around.
type Graken struct {
	URL            string
	Key            string
	Name           string
	ID             string
	Metrics        string
	Email          string
	Role           string
	Team           string
	Group          string
	Folder         string
	AlertEmail     AlertEmail
	AlertSlack     AlertSlack
	AlertPagerDuty AlertPagerDuty
	client         api.ClientInterface
	awsSess        *session.Session
	awsClient      *aws.Config
}

// ParseFlags parses CLI args and sets all the struct data.
// nolint:forbidigo
func (g *Graken) ParseFlags() {
	flag.Usage = func() {
		fmt.Println(`Graken does several things:
 1. Creates CloudWatch Data sources in Grafana.
   - To create a data source, you must pass the -name and -id arguments.
   - The -email argument is optional; it's added to the data source as metadata.
   - The -metrics argument is also optional; use it to add custom metric names.
 2. Creates new Teams in Grafana (bound to an LDAP Group).
   - To create a team you must pass the -team and -group arguments.
 3. Creates Folders in Grafana and adds a Team as an Admin to the Folder.
   - To create a Folder you must pass the -folder and -team arguments.
 4. Creates Notifiation Channels. Like, Alerts.
   - One at a time, pass -alertname "team/service/etc" -alertpdkey "pagerduty api key"
	 - Supports email: -alertemail email@address.com
	 - Supports slack: -alertslack #slack-channel-name

 All arguments may be passed at once and all actions will be invoked! In other
 words, you may create a new Team and a new Folder at the same time. The Team
 will be bound to the provided LDAP Group. Admin privileges on the Folder are
 granted to the team. This tool cannot set permissions on an existing folder.`)
		fmt.Println("\nUsage: graken [-id <id>] [-name <name>] [-team <team>] [-group <group>] [-folder <folder>]")
		flag.PrintDefaults()
	}

	flag.StringVar(&g.ID, "id", "", "AWS Account Number, ex: 42397823232")
	flag.StringVar(&g.Name, "name", "", "AWS Account Name, ex: twitch-central-aws")
	flag.StringVar(&g.Team, "team", "", "Name of the Team being added, or Team that owns a Folder being added.")
	flag.StringVar(&g.Group, "group", "", "LDAP Group name to sync with the Team being added.")
	flag.StringVar(&g.Folder, "folder", "", "Name of the Folder being added.")
	flag.StringVar(&g.URL, "url", defaultGrafanaURL, "Grafana URL")
	flag.StringVar(&g.Metrics, "metrics", "", "Include comma separated custom data source metrics if needed")
	flag.StringVar(&g.Role, "role", "grafana-cloudwatch-read-only", "IAM Role for CloudWatch data source")
	flag.StringVar(&g.Email, "email", "", "Email address is added to a data source for extra data")
	flag.StringVar(&g.AlertSlack.Recipient, "alertslack", "", "Notification Channel: Slack #channel")
	flag.StringVar(&g.AlertEmail.Addresses, "alertemail", "", "Notification Channel: Email Addresses, separate with ;")
	flag.StringVar(&g.AlertPagerDuty.Key, "alertpdkey", "",
		"Notification Channel: PagerDuty API Token. Must pass -alertname")
	flag.StringVar(&g.AlertPagerDuty.name, "alertname", "",
		"Name a new PagerDuty Notification Channel. Must pass -alertpdkey")
	flag.StringVar(&g.Key, "key", os.Getenv("GRAFANA_API_KEY"), "Grafana API Key (env: GRAFANA_API_KEY)")
	flag.Parse()
}

func main() {
	g := &Graken{ //nolint:exhaustivestruct
		awsSess:   session.Must(session.NewSession()),
		awsClient: aws.NewConfig().WithMaxRetries(1),
	}
	g.ParseFlags()

	if err := g.CheckUsage(); err != nil {
		flag.Usage()
		log.Fatalf("ERROR: %v", err)
	} else if err = g.Run(); err != nil {
		log.Fatalf("ERROR: %v", err)
	}
}

// Errors generated by this file.
var (
	ErrInvalidCommand = fmt.Errorf("must provide a combination of arguments to invoke a command")
	ErrMissingAPIURL  = fmt.Errorf("URL or API key missing")
	ErrMissingRole    = fmt.Errorf("role must be set to add a cloudwatch data source")
	ErrMissingIDName  = fmt.Errorf("id and name must both be set to add a cloudwatch data source")
	ErrMissingTeam    = fmt.Errorf("must provide team name when creating a new folder")
	ErrMissingGroup   = fmt.Errorf("must provide LDAP group name to sync with the new team")
	ErrBadPDAlert     = fmt.Errorf("must pass -alertname with -alertpdkey")
)

// CheckUsage makes sure the input data is somewhat valid.
func (g *Graken) CheckUsage() error { // nolint:gocognit
	if g.URL == "" || g.Key == "" {
		return ErrMissingAPIURL
	}

	var cmd bool

	if g.ID != "" || g.Name != "" {
		if g.Role == "" {
			return ErrMissingRole
		}

		if g.ID == "" || g.Name == "" {
			return ErrMissingIDName
		}

		cmd = true
	}

	if g.Folder != "" {
		if g.Team == "" {
			return ErrMissingTeam
		}

		cmd = true
	}

	if g.Team != "" && g.Folder == "" {
		if g.Group == "" {
			return ErrMissingGroup
		}

		cmd = true
	}

	if (g.AlertPagerDuty.Key != "" && g.AlertPagerDuty.name == "") ||
		(g.AlertPagerDuty.name != "" && g.AlertPagerDuty.Key == "") {
		return ErrBadPDAlert
	} else if g.AlertEmail.Addresses != "" ||
		g.AlertSlack.Recipient != "" ||
		g.AlertPagerDuty.Key != "" {
		cmd = true
	}

	if !cmd {
		return ErrInvalidCommand
	}

	return nil
}

// Run the app!
func (g *Graken) Run() (err error) {
	// Grafana Client
	if err = g.GetAPIClient(); err != nil {
		return err
	}

	// Grafana Datasource
	if g.ID != "" {
		if err = g.DoDataSource(); err != nil {
			return err
		}
	}

	var teamID int64
	// Grafana Team
	if g.Group != "" && g.Team != "" {
		if teamID, err = g.DoNewTeam(); err != nil {
			return err
		}
	}

	// Alerts
	if err := g.DoAlerts(); err != nil {
		return err
	}

	// Dashboard Folder
	if g.Folder == "" {
		return nil
	}

	if teamID < 1 {
		teamID, err = g.GetTeamID(g.Team)
	}

	if err == nil {
		_, err = g.DoNewFolder(teamID)
	}

	return err
}

// GetAPIClient creates the client for Grafana.
func (g *Graken) GetAPIClient() error {
	if g.client != nil {
		return nil
	}

	u, err := url.Parse(g.URL)
	if err != nil {
		return fmt.Errorf("parsing url, %s: %w", g.URL, err)
	}

	midway, err := midway.NewClient(nil)
	if err != nil {
		return fmt.Errorf("getting midway cookie: %w", err)
	}

	g.client = api.NewWithClient(&grafanahttp.RESTClient{
		Token:   g.Key,
		BaseURL: u,
		Client:  midway,
	})

	return nil
}
