package midway

import (
	"bufio"
	"net/http"
	"net/http/cookiejar"
	"net/url"
	"os"
	"path/filepath"
	"strconv"
	"strings"
	"time"
)

// Taken from https://code.amazon.com/packages/GoAmzn-MidwayCookie/blobs/mainline/
func midwayCookiePath(path string) (string, error) {
	if path != "" {
		return path, nil
	}

	homeDir, err := os.UserHomeDir()
	if err != nil {
		return "", err
	}

	return filepath.Join(homeDir, ".midway", "cookie"), nil
}

func getMidwayCookies(c *Config) (http.CookieJar, error) {
	path, err := midwayCookiePath(c.GetCookiePath())
	if err != nil {
		return nil, err
	}

	file, err := os.Open(path)
	if err != nil {
		return nil, err
	}

	defer file.Close()

	jar, err := cookiejar.New(nil)
	if err != nil {
		return nil, err
	}

	scanner := bufio.NewScanner(file)
	cookieMap := make(map[*url.URL][]*http.Cookie)

	for scanner.Scan() {
		httpCookie, err := parseCookie(scanner.Text())
		if err != nil {
			return nil, err
		} else if httpCookie == nil {
			continue
		}

		cookieURL, err := getCookieURL(httpCookie)
		if err != nil {
			return nil, err
		}

		if httpCookie.Name == "session" {
			if time.Now().UTC().After(httpCookie.Expires) {
				return nil, err
			}
		}

		cookieMap[cookieURL] = append(cookieMap[cookieURL], httpCookie)
	}

	for url, cookies := range cookieMap {
		jar.SetCookies(url, cookies)
	}

	return jar, nil
}

// getCookieURL will return the parsed URL for the given http cookie structure.
func getCookieURL(httpCookie *http.Cookie) (*url.URL, error) {
	scheme := "https://"
	if httpCookie.HttpOnly {
		scheme = "http://"
	}

	cookieURL, err := url.Parse(scheme + httpCookie.Domain)
	if err != nil {
		return nil, err
	}

	return cookieURL, nil
}

func parseCookie(line string) (*http.Cookie, error) {
	tokens := strings.Split(line, "\t")

	// Strip comments (note the space is essential so that lines starting with '#HttpOnly_' are not read as
	// comments.
	if strings.HasPrefix(tokens[0], "# ") {
		return nil, nil
	}

	// Strip invalid lines. Note that libcurl-like cookie files have 1 cookie per line, each cookie entry
	// exactly 7 elements separated by tabs.
	if len(tokens) != 7 {
		return nil, nil
	}

	// The midway token is HttpOnly, likely so the JavaScript on the page can inspect it.
	httpOnly := false
	cookieDomain := tokens[0]
	if strings.HasPrefix(tokens[0], "#HttpOnly_") {
		cookieDomainParts := strings.Split(tokens[0], "_")
		cookieDomain = cookieDomainParts[1]
		httpOnly = true
	}

	expireTimestamp, err := strconv.ParseInt(tokens[4], 10, 64)
	if err != nil {
		return nil, err
	}

	expireTime := time.Unix(expireTimestamp, 0)
	cookie := &http.Cookie{
		Domain:   cookieDomain,
		Path:     tokens[2],
		HttpOnly: httpOnly,
		Secure:   tokens[3] == "TRUE",
		Expires:  expireTime,
		Name:     tokens[5],
		Value:    tokens[6],
	}

	return cookie, nil
}
