package midway

import (
	"crypto/tls"
	"crypto/x509"
	"errors"
	"net"
	"net/http"
	"strings"
	"time"
)

const (
	defaultTimeout   = 30 * time.Second
	defaultKeepAlive = 30 * time.Second
)

// Config it the optional input data.
type Config struct {
	KeepAlive  time.Duration
	Timeout    time.Duration
	CookiePath string
}

// GetCookiePath returns the configured cookiepath.
func (c *Config) GetCookiePath() string {
	if c == nil {
		return ""
	}

	return c.CookiePath
}

// NewClient returns a midway client ready to use.
func NewClient(c *Config) (*http.Client, error) {
	if c == nil {
		c = &Config{
			KeepAlive: defaultKeepAlive,
			Timeout:   defaultTimeout,
		}
	}

	cookieJar, err := getMidwayCookies(c)
	if err != nil {
		return nil, err
	}

	certs, err := AllCerts()
	if err != nil {
		return nil, err
	}

	return c.getClient(cookieJar, certs), nil
}

func (c *Config) getClient(cookieJar http.CookieJar, certs *x509.CertPool) *http.Client {
	if c.KeepAlive == 0 {
		c.KeepAlive = defaultKeepAlive
	}

	if c.Timeout == 0 {
		c.Timeout = defaultTimeout
	}

	return &http.Client{
		Jar: cookieJar,
		// CheckRedirect: checkRedirect,
		Transport: &http.Transport{
			DialContext: (&net.Dialer{
				Timeout:   c.Timeout,
				KeepAlive: c.KeepAlive,
			}).DialContext,
			TLSHandshakeTimeout: c.Timeout,
			TLSClientConfig: &tls.Config{
				InsecureSkipVerify: false,
				RootCAs:            certs,
			},
		},
	}
}

// Taken from https://git-aws.internal.justin.tv/iduggan/aws-midway-go/blob/master/pkg/midway/client.go#L76
func checkRedirect(req *http.Request, via []*http.Request) error {
	// Redirects sometimes include an explicit port. Including a
	// specific port in the Host header causes some services
	// (isengard-service.amazon.com being one of them) to reject the
	// request.
	req.URL.Host = strings.TrimSuffix(req.URL.Host, ":443")

	if len(via) >= 10 {
		return errors.New("stopped after 10 redirects")
	}
	return nil
}
