package main

import (
	"fmt"
	"log"
	"strings"

	"github.com/nexucis/grafana-go-client/api"
	"github.com/nexucis/grafana-go-client/api/types"
)

// Errors generated by this file.
var (
	ErrUnknownTeam = fmt.Errorf("team not found")
)

// DoNewTeam creates a new team and returns the ID.
func (g *Graken) DoNewTeam() (int64, error) {
	team := &types.CreateOrUpdateTeam{
		Name:  g.Team,
		Email: strings.TrimPrefix(g.Group, "team-") + "@" + ldapEmailSuffix,
	}

	teamID, err := g.client.Teams().Create(team)
	if err != nil {
		return teamID, fmt.Errorf("adding team: %w", err)
	}

	log.Printf("Added New Team: %d:%s (%s)", teamID, team.Name, team.Email)

	return teamID, nil
}

// GetTeamID searches Grafana for Team ID using Team Name.
func (g *Graken) GetTeamID(teamName string) (int64, error) {
	query := api.QueryParameterTeams{Name: teamName} //nolint:exhaustivestruct

	response, err := g.client.Teams().Get(query)
	if err != nil {
		return -1, fmt.Errorf("getting team(s): %w", err)
	}

	if l := len(response.Teams); l == 0 {
		return -1, fmt.Errorf("%w: %s", ErrUnknownTeam, teamName)
	}

	log.Printf("Found ID (%v) for Team: %v", response.Teams[0].ID, teamName)

	return response.Teams[0].ID, nil
}
