package main

import "log"

// Changes holds the list of user additions and deletions from Teams
// that need to happen to sync Grafana with Ldap.
type Changes struct {
	Add UserGroup
	Del UserGroup
}

// ExecChanges adds users to- and deletes users from Grafana Teams
// to keep them in sync with LDAP groups.
func (c *Config) ExecChanges(changes *Changes, teams *GrafanaTeams) error {
	for teamID, users := range changes.Add {
		if err := c.AddTeamMembers(teamID, teams.Name(teamID), users); err != nil {
			return err
		}
	}

	for teamID, users := range changes.Del {
		if err := c.DeleteTeamMembers(teamID, teams.Name(teamID), users); err != nil {
			return err
		}
	}

	return nil
}

// GetChanges returns the list of changes that need to occur to sync Grafana
// with the state of LDAP.
func GetChanges(groupMembers, teamMembers UserGroup, allUsers Users) *Changes {
	changes := &Changes{
		Add: make(UserGroup),
		Del: make(UserGroup),
	}

	for id, groupUsers := range groupMembers {
		teamUsers, ok := teamMembers[id]
		if !ok {
			log.Printf("WARNING: GetDifferences: ID '%v' is missing in team list", id)
			continue //nolint:nlreturn // this shouldn't happen.
		}

		// now we compare the two mapped data sets.
		add, del := GetTeamChanges(teamUsers, groupUsers, allUsers)
		if len(add) > 0 {
			changes.Add[id] = add
		}

		if len(del) > 0 {
			changes.Del[id] = del
		}
	}

	return changes
}

// GetTeamChanges returns two lists showing what to add and delete to make old look like new.
// Only users in the mustContain list are considered to be added to a team.
func GetTeamChanges(old, nEw, mustContain Users) (Users, Users) {
	add := make(Users)
	del := make(Users)

	for name, id := range old {
		if _, ok := nEw[name]; !ok {
			del[name] = id
		}
	}

	for name := range nEw {
		realID, inMustContain := mustContain[name]
		if _, ok := old[name]; inMustContain && !ok {
			add[name] = realID
		}
	}

	return add, del
}
