package main

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"log"
	"net/http"
	"strings"
)

// GrafanaTeams contains a list of every team, but no members.
// Represents the /api/teams/search Grafana API endpoint.
type GrafanaTeams struct {
	TotalCount int           `json:"totalCount"`
	Teams      []GrafanaTeam `json:"teams"`
	Page       int           `json:"page"`
	PerPage    int           `json:"perPage"`
}

// GrafanaTeam is a substruct of the GrafanaTeams interface.
// Represents a portion of the /api/teams/search Grafana API endpoint.
type GrafanaTeam struct {
	ID int `json:"id"`
	//		OrgID       int    `json:"orgId"`
	Name  string `json:"name"`
	Email string `json:"email"`
	//		AvatarURL   string `json:"avatarUrl"`
	//		MemberCount int    `json:"memberCount"`
	//		Permission  int    `json:"permission"`
}

// GrafanaMembers is a list of members in a group.
// Represents the /api/teams/:id/members Grafana API endpoint.
type GrafanaMembers []struct {
	//	OrgID      int           `json:"orgId"`
	//	TeamID     int           `json:"teamId"`
	UserID int `json:"userId"`
	//	AuthModule string        `json:"auth_module"`
	//	Email      string        `json:"email"`
	Login string `json:"login"`
	//	AvatarURL  string        `json:"avatarUrl"`
	//	Labels     []interface{} `json:"labels"`
	//	Permission int           `json:"permission"`
}

// GrafanaUsers is the full user list. It contains only logins, no groups.
// Represents the /api/org/users Grafana API endpoint.
type GrafanaUsers []struct {
	//	OrgID         int       `json:"orgId"`
	UserID int `json:"userId"`
	//	Email         string    `json:"email"`
	//	AvatarURL     string    `json:"avatarUrl"`
	Login string `json:"login"`
	//	Role          string    `json:"role"`
	//	LastSeenAt    time.Time `json:"lastSeenAt"`
	//	LastSeenAtAge string    `json:"lastSeenAtAge"`
}

// GetGrafanaJSON is a helper method to make an HTTP request to Grafana and unmarshal the reply.
func (c *Config) GetGrafanaJSON(uri string, method string, v interface{}, body io.Reader) error {
	ctx, cancel := context.WithTimeout(context.Background(), c.Timeout)
	defer cancel()

	req, err := http.NewRequestWithContext(ctx, method, c.GrafanaURL+uri, body)
	if err != nil {
		return fmt.Errorf("creating request: %w", err)
	}

	req.Header.Set("Content-Type", "application/json")
	req.Header.Set("Authorization", "Bearer "+c.GrafanaKey)

	resp, err := (&http.Client{}).Do(req)
	if err != nil {
		return fmt.Errorf("making request: %w", err)
	}
	defer resp.Body.Close()

	data, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return fmt.Errorf("reading response: %w", err)
	}

	if err := json.Unmarshal(data, v); err != nil {
		return fmt.Errorf("unmarshaling: %w", err)
	}

	return nil
}

// GetGrafanaUsers returns a list of every user in Grafana.
func (c *Config) GetGrafanaUsers() (Users, error) {
	users := Users{}
	gu := &GrafanaUsers{}

	err := c.GetGrafanaJSON("/api/org/users", "GET", gu, nil)

	for _, user := range *gu {
		users[user.Login] = user.UserID
	}

	return users, err
}

// GetGrafanaTeams returns a list of all the teams in Grafana.
func (c *Config) GetGrafanaTeams() (*GrafanaTeams, error) {
	teams := &GrafanaTeams{}

	err := c.GetGrafanaJSON("/api/teams/search", "GET", teams, nil)
	if err != nil {
		return teams, err
	}

	return teams, nil
}

// DeleteTeamMembers removes a list of Users from a Team.
func (c *Config) DeleteTeamMembers(teamID int, teamName string, users Users) error {
	for username, uid := range users {
		v := new(interface{}) // we don't look at the reply.

		log.Printf("Grafana: Deleting %d:%s from %d:%s", uid, username, teamID, teamName)

		err := c.GetGrafanaJSON(fmt.Sprintf("/api/teams/%d/members/%d", teamID, uid), "DELETE", v, nil)
		if err != nil {
			return fmt.Errorf("%v %w", v, err)
		}
	}

	return nil
}

// AddTeamMembers adds a list of Users to a Team.
func (c *Config) AddTeamMembers(teamID int, teamName string, users Users) error {
	for username, uid := range users {
		v := new(interface{}) // we don't look at the reply.

		log.Printf("Grafana: Adding %d:%s to %d:%s", uid, username, teamID, teamName)

		err := c.GetGrafanaJSON(fmt.Sprintf("/api/teams/%d/members", teamID),
			"POST", v, bytes.NewBufferString(fmt.Sprintf(`{"UserId": %d}`, uid)))
		if err != nil {
			return fmt.Errorf("%v %w", v, err)
		}
	}

	return nil
}

// GetGrafanaMembers returns the members for the list of provided Grafana teams.
func (c *Config) GetGrafanaMembers(teams []GrafanaTeam) (UserGroup, error) {
	members := UserGroup{}

	for _, team := range teams {
		resp := GrafanaMembers{}

		err := c.GetGrafanaJSON(fmt.Sprintf("/api/teams/%d/members", team.ID), "GET", &resp, nil)
		if err != nil {
			return nil, err
		}

		members[team.ID] = Users{}

		for _, member := range resp {
			members[team.ID][member.Login] = member.UserID
		}
	}

	return members, nil
}

// Filter returns a subset of the available Grafana Teams.
// Only returns teams that have an email with the word `filter` in it.
func (g *GrafanaTeams) Filter(filter string) (groups []GrafanaTeam) {
	for _, team := range g.Teams {
		if strings.Contains(team.Email, filter) {
			groups = append(groups, team)
		}
	}

	return
}

// Name returns the name for a team by ID.
func (g *GrafanaTeams) Name(id int) string {
	for _, team := range g.Teams {
		if team.ID == id {
			return team.Name
		}
	}

	return ""
}
