package main

import (
	"crypto/tls"
	"fmt"
	"log"
	"strings"

	ldap "github.com/go-ldap/ldap/v3"
)

// UserGroup holds groups->users or teams->users.
type UserGroup map[int]Users

// Users is a map of username->ID.
type Users map[string]int

// teamQuery is used to send data into the worker pool.
type teamQuery struct {
	TeamID int
	Team   string
	Group  string
}

// GetGroupsMembers returns all the uids that belong to a list of LDAP groups
// parsed out of grafana team email address.
// The email address is expected to be: group-name@ldap.twitch.a2z.com
// The Team's LDAP group name becomes: team-group-name
// 'teams' is a list of Grafana teams from the Grafana Teams API.
func (c *Config) GetGroupsMembers(teams []GrafanaTeam) UserGroup {
	found := make(UserGroup)
	replies := make(chan *UserGroup, len(teams))
	queries := make(chan *teamQuery, len(teams))

	threads := c.Threads
	if len(teams) < threads {
		threads = len(teams)
	}

	log.Printf("Connecting to ldaps://%s:636 using %d threads.", c.LDAPServer, threads)

	// Make multiple LDAP connections to send queries to.
	for w := 1; w <= threads; w++ {
		go c.ldapWorker(queries, replies)
	}

	for _, team := range teams {
		// We only want the stuff before the @.
		groupName := strings.Split(team.Email, "@")[0]

		// Add team- prefix to group name if it's missing.
		if !strings.HasPrefix(groupName, "team-") {
			groupName = "team-" + groupName
		}

		queries <- &teamQuery{TeamID: team.ID, Group: groupName, Team: team.Name}
	}

	close(queries)

	// Collect all the results, and save them.
	for a := 0; a < len(teams); a++ {
		team := <-replies
		if team == nil {
			// If the ldap query had an error, reply is nil, do not record!
			continue
		}

		for teamID, users := range *team {
			found[teamID] = users
		}
	}

	return found
}

func (c *Config) ldapWorker(queries <-chan *teamQuery, results chan<- *UserGroup) {
	l, err := ldap.DialTLS("tcp", c.LDAPServer+":636", &tls.Config{ServerName: c.LDAPServer})
	if err != nil {
		log.Printf("ERROR: ldap.DialTLS: %v", err)
		return
	}
	defer l.Close()

	for query := range queries {
		log.Printf("Querying LDAP Group '%s' for Team: %s", query.Group, query.Team)

		users, err := GetLDAPGroupUsers(query.Group, l, c.UsersDN, c.GroupDN)
		if err != nil {
			log.Printf("ERROR: ldapConn.Search: %v", err)
			results <- nil

			continue
		}

		if len(users) < 1 {
			log.Printf("WARNING: Group does not have members, searched: %v", query.Group)
		}

		results <- &UserGroup{query.TeamID: users}
	}
}

// GetLDAPGroupUsers takes an ldap Client and a group name.
// Returns the group member uids from LDAP.
func GetLDAPGroupUsers(group string, ldapConn ldap.Client, usersDN, groupDN string) (Users, error) {
	searchRequest := ldap.NewSearchRequest(usersDN,
		ldap.ScopeWholeSubtree, ldap.NeverDerefAliases, 0, 0, false,
		fmt.Sprintf("(memberOf=cn=%s,%s)", group, groupDN),
		[]string{"uid"}, nil)

	sr, err := ldapConn.Search(searchRequest)
	if err != nil {
		return nil, fmt.Errorf("searching ldap: %w", err)
	}

	users := Users{}

	for _, entry := range sr.Entries {
		for _, attr := range entry.GetAttributeValues("uid") {
			users[attr] = 0
		}
	}

	return users, nil
}
