package main

import (
	"flag"
	"fmt"
	"log"
	"os"
	"strconv"
	"time"

	"github.com/aws/aws-lambda-go/lambda"
	ldap "github.com/go-ldap/ldap/v3"
)

const (
	defaultLdapServer  = "ldap.twitch.a2z.com"
	defaultGrafanaURL  = "https://grafana.xarth.tv"
	defaultLdapGrupDN  = "ou=Groups,dc=justin,dc=tv"
	defaultLdapUserDN  = "ou=Users,dc=justin,dc=tv"
	defaultFilterTerm  = "ldap.twitch"
	defaultLDAPThreads = 11
)

// Config contains our running configuration.
type Config struct {
	LDAPServer string
	UsersDN    string
	GroupDN    string
	Filter     string
	GrafanaURL string
	GrafanaKey string
	Threads    int
	Lambda     bool
	Timeout    time.Duration
}

// Get the input data and go Work.
func main() {
	c := &Config{}
	c.ParseFlags()
	if c.Lambda {
		lambda.Start(c.Work)
		return
	}
	if err := c.Work(); err != nil {
		log.Fatal(err)
	}
}

// ParseFlags turns intput cli flags into structured config data.
func (c *Config) ParseFlags() {
	lambdaEnv, _ := strconv.ParseBool(os.Getenv("LAMBDA"))

	// If you set LDAP_THREADS to a string, the error gets swallowed and threads is set to default.
	threads, err := strconv.Atoi(os.Getenv("LDAP_THREADS"))
	if err != nil && os.Getenv("LDAP_THREADS") != "" {
		log.Println("WARNING: LDAP_THREADS:", err)
	}

	flag.BoolVar(&c.Lambda, "lambda", lambdaEnv, "Invoke lambda (instead of this, use env var: LAMBDA=true)")
	flag.StringVar(&c.LDAPServer, "ldap", defaultLdapServer, "LDAP server hostname")
	flag.StringVar(&c.GroupDN, "group", defaultLdapGrupDN, "Group DN to search")
	flag.StringVar(&c.UsersDN, "users", defaultLdapUserDN, "Users DN to search")
	flag.StringVar(&c.Filter, "filter", defaultFilterTerm, "Sync teams with this string in their email")
	flag.DurationVar(&c.Timeout, "timeout", time.Minute, "Grafana timeout")
	flag.StringVar(&c.GrafanaKey, "key", os.Getenv("GRAFANA_API_KEY"), "Grafana API key")
	flag.StringVar(&c.GrafanaURL, "url", os.Getenv("GRAFANA_URL"), "Grafana URL")
	flag.IntVar(&c.Threads, "threads", threads, "Count of threads when querying with LDAP")
	flag.Parse()

	if c.Threads == 0 {
		c.Threads = defaultLDAPThreads
	}

	if c.GrafanaURL == "" {
		c.GrafanaURL = defaultGrafanaURL
	}
}

// Work does all the data collection and updates:
// Collects all users, all teams, and all team members from Grafana.
// Filters only specific teams from Grafana; those containing 'ldap'.
// Then collects a list of all group members from LDAP for the filtered Grafana Team list.
// The data from both systems is compared, and two lists of changes for each team is created.
// Finally the changes are executed against Grafana to bring the Teams in sync with LDAP.
func (c *Config) Work() error {
	start := time.Now()

	log.SetFlags(log.Lmicroseconds | log.LstdFlags)
	log.Println("Collecting Grafana Users from", c.GrafanaURL)

	grafanaUsers, err := c.GetGrafanaUsers()
	if err != nil {
		return fmt.Errorf("GetGrafanaUsers: %w", err)
	}

	log.Printf("Collecting Grafana Teams (found %d users)", len(grafanaUsers))

	teams, err := c.GetGrafanaTeams()
	if err != nil {
		return fmt.Errorf("GetGrafanaTeams: %w", err)
	}

	filteredTeams := teams.Filter(c.Filter)
	log.Printf("Collecting Grafana Team Members; %d total teams, filtered %d", len(teams.Teams), len(filteredTeams))

	teamsMembers, err := c.GetGrafanaMembers(filteredTeams)
	if err != nil {
		return fmt.Errorf("GetGrafanaMembers: %w", err)
	}

	/* LDAP calls come next. */
	// Set the global LDAP timeout.
	ldap.DefaultTimeout = c.Timeout
	groupMembers := c.GetGroupsMembers(filteredTeams)
	changes := GetChanges(groupMembers, teamsMembers, grafanaUsers)

	log.Printf("Calculating Team/Group Differences; Groups: %d, Teams: %d", len(groupMembers), len(teamsMembers))

	if len(changes.Add) < 1 && len(changes.Del) < 1 {
		log.Printf("No changes needed! Elapsed: %v", time.Since(start).Round(time.Millisecond))
		return nil
	}

	log.Printf("Executing Grafana Changes; Add: %d, Del: %d", len(changes.Add), len(changes.Del))

	err = c.ExecChanges(changes, teams)

	log.Printf("Changes Complete! Elapsed: %v", time.Since(start).Round(time.Millisecond))

	return err
}
