terraform {
  backend "s3" {
    bucket  = "twitch-grafana-dev"
    key     = "tfstate/systems/systems-terraform/development/grafana-dev/db/terraform.tfstate"
    region  = "us-west-2"
    profile = "twitch-grafana-dev"
  }
}

provider "aws" {
  profile = var.account
  alias   = "twitch-grafana-dev"
  region  = var.region
}

# invoke the module for tagging
module "taglib" {
  providers = {
    aws = aws.twitch-grafana-dev
  }

  source      = "git::git+ssh://git@git.xarth.tv/terraform-modules/taglib?ref=v0.4"
  name        = "grafana-${var.env}"
  owner       = "systems"
  service     = "grafana"
  srctag      = "ghe/systems/systems-terraform:${var.env}/grafana-dev"
  environment = var.env
}

# DB infra
resource "aws_db_subnet_group" "grafana" {
  provider   = aws.twitch-grafana-dev
  name       = "grafana"
  subnet_ids = var.subnet_ids
  tags       = module.taglib.tags
}

data "aws_ssm_parameter" "db_password" {
  provider = aws.twitch-grafana-dev
  name     = "/twitch-grafana-dev/db-password-plain.txt"
}

resource "aws_rds_cluster" "grafana_postgresql" {
  provider                = aws.twitch-grafana-dev
  cluster_identifier      = "grafana-aurora-cluster-${var.env}"
  engine                  = "aurora-postgresql"
  engine_mode             = "serverless"
  availability_zones      = ["us-west-2a", "us-west-2b", "us-west-2c"]
  db_subnet_group_name    = aws_db_subnet_group.grafana.name
  vpc_security_group_ids  = var.vpc_security_group_ids
  skip_final_snapshot     = true
  database_name           = "grafana"
  master_username         = "grafana"
  master_password         = data.aws_ssm_parameter.db_password.value
  backup_retention_period = 5
  kms_key_id              = "arn:aws:kms:us-west-2:109561199089:key/65ce811c-ddc3-4443-a3af-80f002cf4e1c" # from prod.
  tags                    = module.taglib.tags

  scaling_configuration {
    auto_pause               = true
    max_capacity             = 64
    min_capacity             = 2
    seconds_until_auto_pause = 300
    timeout_action           = "ForceApplyCapacityChange"
  }
}

# aurora.grafana.twitch.a2z.com - this doesn't work right for some reason.
# manually point sql.grafana.twitch.a2z.com to the VPC endpoint created by aws_rds_cluster
resource "aws_route53_record" "grafana_postgresql" {
  provider = aws.twitch-grafana-dev
  zone_id  = var.r53_zone_id
  name     = "aurora"
  type     = "A"

  alias {
    name                   = aws_rds_cluster.grafana_postgresql.endpoint
    zone_id                = aws_rds_cluster.grafana_postgresql.hosted_zone_id
    evaluate_target_health = "false"
  }
}

output "db" {
  value = [aws_rds_cluster.grafana_postgresql.endpoint, aws_route53_record.grafana_postgresql.fqdn]
}
