data "archive_file" "delete_db" {
  type        = "zip"
  output_path = "delete_old_db_dest_aurora.zip"
  source {
    content = file(
      "deploy_db_from_snapshot_dest_aurora/snapshots_tool_utils.py",
    )
    filename = "snapshots_tool_utils.py"
  }
  source {
    content  = file("delete_old_db_dest_aurora/lambda_handler.py")
    filename = "lambda_handler.py"
  }
}

resource "aws_lambda_function" "aurora_lambdas_delete" {
  provider         = aws.twitch-grafana-dev
  function_name    = "delete-old-aurora-snapshot-db"
  filename         = data.archive_file.delete_db.output_path
  source_code_hash = data.archive_file.delete_db.output_base64sha256
  role             = aws_iam_role.aurora_lambdas.arn
  handler          = "lambda_handler.lambda_handler"
  runtime          = "python3.6"
  timeout          = 30
  environment {
    variables = {
      LOG_LEVEL = "INFO"
    }
  }
  reserved_concurrent_executions = 1
}

resource "aws_cloudwatch_event_rule" "every_day_at_3" {
  provider            = aws.twitch-grafana-dev
  name                = "delete-old-aurora-db-daily"
  description         = "Deletes the old production aurora db every day."
  schedule_expression = "cron(0 10 * * ? *)" # 3:00 AM PST
}

resource "aws_sfn_state_machine" "aurora_lambdas_delete" {
  provider   = aws.twitch-grafana-dev
  name       = "delete-old-aurora-snapshot-db"
  role_arn   = aws_iam_role.aurora_lambdas.arn
  definition = <<EOF
{
  "Comment": "DeleteOld for Aurora prod database",
  "StartAt": "DeleteOldProdDatabase",
  "States": {
    "DeleteOldProdDatabase": {
      "Type":     "Task",
      "Resource": "${aws_lambda_function.aurora_lambdas_delete.arn}",
      "Retry":    [
        {
          "ErrorEquals":     ["States.ALL"],
          "IntervalSeconds": 600,
          "MaxAttempts":     10,
          "BackoffRate":     1
       }
     ],
     "End": true
    }
  }
}
EOF

}

resource "aws_cloudwatch_event_target" "aurora_lambdas_delete" {
  provider = aws.twitch-grafana-dev
  rule     = aws_cloudwatch_event_rule.every_day_at_3.name
  arn      = aws_sfn_state_machine.aurora_lambdas_delete.id
  role_arn = aws_iam_role.aurora_lambdas.arn
}

